import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { EVENT, VIDEO_CHANNEL_CHANNEL } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, createAlias, isObjectId
} from 'src/core';
import { ChannelDto, ProfileDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { ChannelCreatePayload, ChannelUpdatePayload } from 'src/payloads/channel';
import { Channel, ChannelDocument } from 'src/schemas';

import { ReactionService } from '../reaction';

@Injectable()
export class ChannelService {
  constructor(
    @InjectModel(Channel.name) private readonly ChannelModel: Model<ChannelDocument>,
    private readonly fileService: FileService,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly queueService: QueueMessageService
  ) { }

  public async generateSlug(title: string, id?: string | ObjectId) {
    // consider if need unique slug with post type
    const slug = createAlias(title);
    const query = { slug } as any;
    if (id) {
      query._id = { $ne: id };
    }
    const count = await this.ChannelModel.countDocuments(query);
    if (!count) {
      return slug;
    }

    return this.generateSlug(`${slug}1`, id);
  }

  public async create(payload: ChannelCreatePayload, current: ProfileDto): Promise<ChannelDto> {
    const slug = await this.generateSlug(payload?.slug || payload.name);
    const channel = await this.ChannelModel.create({
      ...payload,
      slug,
      createdBy: current._id
    });

    return plainToInstance(ChannelDto, channel.toObject());
  }

  public async findByIdOrSlug(id: string | ObjectId): Promise<ChannelDocument> {
    const query = id instanceof ObjectId || isObjectId(id)
      ? { _id: id }
      : { slug: id };
    return this.ChannelModel.findOne(query);
  }

  public async update(id: string | ObjectId, payload: ChannelUpdatePayload): Promise<ChannelDto> {
    const channel = await this.ChannelModel.findById(id);
    if (!channel) throw new EntityNotFoundException();
    const slug = await this.generateSlug(payload?.slug || payload.name, channel._id);
    merge(channel, { ...payload, slug });
    channel.updatedAt = new Date();
    await channel.save();
    return plainToInstance(ChannelDto, channel.toObject());
  }

  public async updateSubscriptionStats(channelId: string | ObjectId, value: number) {
    const channel = await this.findByIdOrSlug(channelId);
    if (!channel) return;

    await this.ChannelModel.updateOne({ _id: channelId }, {
      $inc: { 'stats.subscribers': value }
    });
  }

  public async delete(id: string | ObjectId): Promise<any> {
    const channel = await this.ChannelModel.findById(id);
    if (!channel) throw new EntityNotFoundException();

    await channel.deleteOne();
    const dto = plainToInstance(ChannelDto, channel.toObject());
    await this.queueService.publish(VIDEO_CHANNEL_CHANNEL, {
      eventName: EVENT.DELETED,
      data: dto
    });
    return true;
  }

  public async getDetails(id: string | ObjectId, user?: ProfileDto) {
    const channel = await this.findByIdOrSlug(id);
    if (!channel) throw new EntityNotFoundException();

    const dto = plainToInstance(ChannelDto, channel.toObject());
    if (user && user._id) {
      const { isFollowed } = await this.reactionService.getReactionInfo(dto._id, user);
      dto.isFollowed = isFollowed;
    }
    if (channel.imageId) {
      const image = await this.getFileUploadedInfo(channel.imageId.toString());
      if (image.url) dto.setImageUrl(image);
    }
    return dto;
  }

  public async signUploadImageUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'channel-avatar',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string, authenticated = false, expiresIn = 3600) {
    const file = await this.fileService.getPubicInfo({
      fileId,
      authenticated,
      expiresIn
    });
    if (!file) {
      throw new EntityNotFoundException();
    }
    return file;
  }

  public async handleUpdateVideoStats(channelId: string | ObjectId, value: number) {
    return this.ChannelModel.updateOne({
      _id: channelId
    }, {
      $set: {
        'stats.videos': value
      }
    });
  }
}
