import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model, SortOrder } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import { PageableData } from 'src/core';
import { ChannelDto, ProfileDto } from 'src/dtos';
import { FileService } from 'src/file-module/services';
import { ChannelSearchPayload } from 'src/payloads/channel';
import { Channel, ChannelDocument } from 'src/schemas';

import { ReactionService } from '../reaction';
import { VideoService } from '../video';

@Injectable()
export class ChannelSearchService {
  constructor(
    @InjectModel(Channel.name) private readonly ChannelModel: Model<ChannelDocument>,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly fileService: FileService,
    @Inject(forwardRef(() => VideoService))
    private readonly videoService: VideoService
  ) { }

  public async populateChannelData(data: ChannelDocument[], user?: ProfileDto): Promise<ChannelDto[]> {
    const channelIds = data.map((c) => c._id.toString());
    const imageIds = data.filter((c) => !!c.imageId).map((c) => c.imageId.toString());

    const [reactions, images, videoAggs] = await Promise.all([
      user ? this.reactionService.findByQuery({
        objectId: {
          $in: channelIds
        },
        createdBy: user._id
      }) : [],
      this.fileService.getFilesPublicInfo({ fileIds: imageIds, authenticated: true }),
      this.videoService.countTotalVideoViewsByChannelId(channelIds)
    ]);

    return data.map((c) => {
      const channel = plainToInstance(ChannelDto, c);
      const follow = reactions.find((l) => l.objectId.toString() === c._id.toString() && l.action === REACTION_ACTION.FOLLOW);
      channel.isFollowed = !!follow;

      if (c.imageId) {
        const image = images[c.imageId.toString()];
        channel.setImageUrl(image);
      }

      const videoAgg = videoAggs.find((item) => `${item._id}` === `${c._id}`);
      if (videoAgg) {
        channel.setTotalVideoView(videoAgg);
      }
      return channel;
    });
  }

  public async advancedSearch(payload: ChannelSearchPayload): Promise<PageableData<ChannelDto>> {
    const query: FilterQuery<ChannelDocument> = {};

    if (payload.q) {
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          name: { $regex: regexp }
        },
        {
          slug: { $regex: regexp }
        }
      ];
    }

    if (payload.status) {
      query.status = payload.status;
    }

    const [data, total] = await Promise.all([
      this.ChannelModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.ChannelModel.countDocuments(query)
    ]);

    return {
      data: data.map((channel) => plainToInstance(ChannelDto, channel)),
      total
    };
  }

  public async search(payload: ChannelSearchPayload, user?: ProfileDto): Promise<PageableData<ChannelDto>> {
    const query: FilterQuery<ChannelDocument> = {
      status: 'active'
    };

    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        }
      ];
    }

    const sort: { [key: string]: SortOrder } = {};
    if (payload.sortBy === 'most_videos') {
      sort['stats.videos'] = 'desc';
    }
    sort.createdAt = 'desc';

    const [data, total] = await Promise.all([
      this.ChannelModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.ChannelModel.countDocuments(query)
    ]);

    return {
      data: await this.populateChannelData(data, user),
      total
    };
  }


  public async findByIds(channelIds: string[] | ObjectId[], user?: ProfileDto) {
    const data = await this.ChannelModel.find({ _id: { $in: channelIds }, status: 'active' }).lean().exec();
    return this.populateChannelData(data, user);
  }
}
