import { Injectable, Logger } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import { ChannelDto, ProfileDto } from 'src/dtos';
import { IGetReactionObjectInfoResponse, IUpdateObjectStats } from 'src/interfaces';
import { ReactionUpdateStatsPayload } from 'src/payloads';
import { Channel, ChannelDocument } from 'src/schemas';

import { ChannelSearchService } from './channel-search.service';
import { AbstractReaction } from '../reaction';

@Injectable()
export class ChannelReactionService extends AbstractReaction {
  private logger = new Logger(ChannelReactionService.name);

  constructor(
    @InjectModel(Channel.name) private readonly ChannelModel: Model<ChannelDocument>,
    private readonly channelSearchService: ChannelSearchService
  ) {
    super();
  }

  public async getReactionObjectInfo(ids: string[], user?: ProfileDto): Promise<IGetReactionObjectInfoResponse<ChannelDto>> {
    const data = await this.ChannelModel.find({ _id: { $in: ids } })
      .lean();

    return {
      items: await this.channelSearchService.populateChannelData(data, user)// TO-DO need to check current user subscribed channel
    };
  }

  public async handleUpdateStats(objectId: string | ObjectId, payload: ReactionUpdateStatsPayload): Promise<IUpdateObjectStats> {
    try {
      const channel = await this.ChannelModel.findById(objectId);
      if (!channel) {
        return;
      }
      const { action, value } = payload;
      let fieldToUpdate = '';
      switch (action) {
        case REACTION_ACTION.LIKE:
          fieldToUpdate = 'stats.likes';
          break;
        case REACTION_ACTION.FOLLOW:
          fieldToUpdate = 'stats.followers';
          break;
        default:
          break;
      }
      if (!fieldToUpdate) return;

      const updateQuery: FilterQuery<ChannelDocument> = {
        _id: objectId
      };
      if (value < 0) {
        updateQuery[fieldToUpdate] = { $gte: 1 };
      }
      await this.ChannelModel.updateOne(updateQuery, {
        $inc: {
          [fieldToUpdate]: value
        }
      });
    } catch (e) {
      this.logger.error(e);
    }
  }
}
