import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { CATEGORY_CHANNEL, CATEGORY_GROUP, EVENT } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, createAlias, isObjectId
} from 'src/core';
import { CategoryDto } from 'src/dtos';
import { IPublicFileInfo } from 'src/file-module/interfaces';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { CategoryCreatePayload, CategoryUpdatePayload } from 'src/payloads/category';
import { Category, CategoryDocument } from 'src/schemas';

@Injectable()
export class CategoryService {
  constructor(
    @InjectModel(Category.name) private readonly CategoryModel: Model<CategoryDocument>,
    private readonly fileService: FileService,
    private readonly queueMessageService: QueueMessageService
  ) { }

  public async findById(id: string | ObjectId): Promise<CategoryDto> {
    const item = await this.CategoryModel.findById(id);
    if (!item) return null;

    return plainToInstance(CategoryDto, item.toObject());
  }

  public async generateSlug(title: string, group: string, id?: string | ObjectId) {
    // consider if need unique slug with post type
    const slug = createAlias(title);
    const query = { slug, group } as any;
    if (id) {
      query._id = { $ne: id };
    }
    const count = await this.CategoryModel.countDocuments(query);
    if (!count) {
      return slug;
    }

    return this.generateSlug(`${slug}1`, group, id);
  }

  public async create(payload: CategoryCreatePayload): Promise<CategoryDto> {
    const group = payload.group || CATEGORY_GROUP.VIDEO;
    const slug = await this.generateSlug(payload?.slug || payload.title, group);

    const category = await this.CategoryModel.create({
      ...payload,
      group,
      slug,
      createdAt: new Date(),
      updatedAt: new Date()
    });
    await this.queueMessageService.publish(CATEGORY_CHANNEL, {
      eventName: EVENT.CREATED,
      data: plainToInstance(CategoryDto, category.toObject())
    });
    return plainToInstance(CategoryDto, category.toObject());
  }

  public async update(id: string | ObjectId, payload: CategoryUpdatePayload): Promise<CategoryDto> {
    const category = await this.CategoryModel.findById(id);
    if (!category) throw new EntityNotFoundException();

    const group = payload.group || CATEGORY_GROUP.VIDEO;
    const slug = await this.generateSlug(payload?.slug || payload.title, group, category._id);

    await this.CategoryModel.updateOne({ _id: category._id }, {
      ...payload,
      group,
      slug,
      updatedAt: new Date()
    });
    const newData = await this.CategoryModel.findById(category._id);
    return plainToInstance(CategoryDto, newData.toObject());
  }

  public async delete(id: string | ObjectId): Promise<boolean> {
    const category = await this.CategoryModel.findById(id);
    if (!category) throw new EntityNotFoundException();

    await category.deleteOne();
    await this.queueMessageService.publish(CATEGORY_CHANNEL, {
      eventName: EVENT.DELETED,
      data: plainToInstance(CategoryDto, category.toObject())
    });
    return true;
  }

  public async getDetails(id: string | ObjectId) {
    const query = isObjectId(id.toString()) ? { _id: id } : { slug: id.toString() };
    const category = await this.CategoryModel.findOne(query);
    if (!category) throw new EntityNotFoundException();
    const dto = plainToInstance(CategoryDto, category.toObject());
    if (category.posterId) {
      const posters = await this.fileService.getFilesPublicInfo({
        fileIds: [category.posterId.toString()]
      });
      if (posters && posters[category.posterId.toString()]) {
        dto.setPosterUrl(posters[category.posterId.toString()]);
      }
    }
    // TODO - check for posted if any
    return dto;
  }

  public async signUploadPosterUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'category-poster',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string, authenticated = false, expiresIn = 3600) {
    const file = await this.fileService.getPubicInfo({
      fileId,
      authenticated,
      expiresIn
    });
    if (!file) {
      throw new EntityNotFoundException();
    }
    return file as IPublicFileInfo;
  }

  public async createCategoryViaVideoImport(categories: string) {
    const arrayCatNames = categories.includes(',') ? categories.split(',') : categories.split(';');
    const ids = [];
    if (arrayCatNames.length > 0) {
      await arrayCatNames.reduce(async (lp, cat) => {
        await lp;
        const slug = createAlias(cat);
        let category = await this.CategoryModel.findOne({
          slug
        });
        if (!category) {
          category = await this.CategoryModel.create({
            title: cat,
            slug,
            group: CATEGORY_GROUP.VIDEO,
            status: 'active'
          });
          await this.queueMessageService.publish(CATEGORY_CHANNEL, {
            eventName: EVENT.CREATED,
            data: plainToInstance(CategoryDto, category.toObject())
          });
        }
        ids.push(category._id);
        return Promise.resolve();
      }, Promise.resolve());
    }
    return ids;
  }

  public async handleUpdateVideoStats(categoryId: string | ObjectId, value: number) {
    return this.CategoryModel.updateOne({
      _id: categoryId
    }, {
      $set: {
        'stats.videos': value
      }
    });
  }
}
