import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model, SortOrder } from 'mongoose';

import { PageableData } from 'src/core';
import { CategoryDto } from 'src/dtos';
import { FileService } from 'src/file-module/services';
import { CategorySearchRequest } from 'src/payloads/category';
import { CategoryDocument, Category } from 'src/schemas';


@Injectable()
export class CategorySearchService {
  constructor(
    @InjectModel(Category.name) private readonly CategoryModel: Model<CategoryDocument>,
    private readonly fileService: FileService
  ) { }

  public async populateData(data: CategoryDocument[]): Promise<CategoryDto[]> {
    const posterIds = data.filter((c) => !!c.posterId).map((c) => c.posterId.toString());
    const files = await this.fileService.getFilesPublicInfo({ fileIds: posterIds, authenticated: true });

    return data.map((category) => {
      const cat = plainToInstance(CategoryDto, category);
      if (cat.posterId) {
        const poster = files[category.posterId.toString()];
        cat.setPosterUrl(poster);
      }
      return cat;
    });
  }

  public async advancedSearch(payload: CategorySearchRequest): Promise<PageableData<CategoryDto>> {
    const query: FilterQuery<CategoryDocument> = {};

    if (payload.q) {
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          title: { $regex: regexp }
        },
        {
          slug: { $regex: regexp }
        }
      ];
    }

    ['group', 'status'].forEach((q) => {
      if (payload[q]) query[q] = payload[q];
    });

    const sort = {
      [payload.sortBy || 'ordering']: payload.sort || 1,
      createdAt: 1
    } as Record<string, any>;

    const [data, total] = await Promise.all([
      this.CategoryModel.find(query)
        .collation({ locale: 'en' })
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.CategoryModel.countDocuments(query)
    ]);

    return {
      data: await this.populateData(data),
      total
    };
  }

  public async search(req: CategorySearchRequest): Promise<PageableData<CategoryDto>> {
    const query: FilterQuery<CategoryDocument> = {
      status: 'active'
    };

    if (req.q) {
      const searchValue = { $search: `"${req.q.toLowerCase()}"` };
      const regexp = new RegExp(
        req.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          title: { $regex: regexp }
        },
        {
          slug: { $regex: regexp }
        }
      ];
    }

    ['group'].forEach((q) => {
      if (req[q]) query[q] = req[q];
    });

    let sort: { [key: string]: SortOrder } = {};
    if (req.sortBy === 'new') {
      sort.createdAt = 'desc';
    } else if (req.sortBy === 'most_videos') {
      sort['stats.videos'] = 'desc';
    } else {
      sort = {
        ordering: 'desc',
        createdAt: 'desc'
      };
    }

    const [data, total] = await Promise.all([
      this.CategoryModel.find(query)
        .sort(sort)
        .limit(req.limit)
        .skip(req.offset)
        .lean()
        .exec(),
      this.CategoryModel.countDocuments(query)
    ]);

    return {
      data: await this.populateData(data),
      total
    };
  }

  public async findByIds(categoryIds: string[] | ObjectId[]) {
    const data = await this.CategoryModel.find({ _id: { $in: categoryIds }, status: 'active' }).lean().exec();
    return this.populateData(data);
  }
}
