import * as crypto from 'crypto';

import {
  HttpException,
  Injectable
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import * as jwt from 'jsonwebtoken';
import { Model } from 'mongoose';

import { EntityNotFoundException } from 'src/core';
import { isEmail, randomString } from 'src/core/helpers';
import { LoginPasswordPayload } from 'src/payloads';
import { Auth, ForgotPassword, ForgotPasswordDoument } from 'src/schemas';

import { MailService } from './mailer';
import { ProfileService } from './profile/profile.service';
import {
  AuthCreateDto, AuthDto, AuthUpdateDto, ProfileDto
} from '../dtos';


@Injectable()
export class AuthService {
  constructor(
    private configService: ConfigService,
    private readonly profileService: ProfileService,
    private readonly mailSerivce: MailService,
    @InjectModel(Auth.name) private readonly AuthModel: Model<Auth>,
    @InjectModel(ForgotPassword.name) private readonly ForgotPasswordModel: Model<ForgotPasswordDoument>
  ) { }

  /**
   * generate password salt
   * @param byteSize integer
   */
  public generateSalt(byteSize = 16): string {
    return crypto.randomBytes(byteSize).toString('base64');
  }

  public encryptPassword(pw: string, salt: string): string {
    const defaultIterations = 10000;
    const defaultKeyLength = 64;

    return crypto.pbkdf2Sync(pw, salt, defaultIterations, defaultKeyLength, 'sha1').toString('base64');
  }

  public async createAuthPassword(data: AuthCreateDto): Promise<AuthDto> {
    const salt = this.generateSalt();
    const newVal = this.encryptPassword(data.value, salt);
    // avoid admin update
    // TODO - should listen via user event?
    let auth = await this.AuthModel.findOne({
      type: 'password',
      profileId: data.profileId
    });
    if (!auth) {
      // eslint-disable-next-line new-cap
      auth = new this.AuthModel({
        type: 'password',
        profileId: data.profileId
      });
    }

    auth.salt = salt;
    auth.value = newVal;
    auth.key = data.key;

    const res = await auth.save();
    return new AuthDto(res);
  }

  public async updateAuthPassword(data: AuthUpdateDto) {
    return this.createAuthPassword(data);
  }

  public verifyPassword(pw: string, auth: Auth): boolean {
    if (!pw || !auth || !auth.salt) {
      return false;
    }
    return this.encryptPassword(pw, auth.salt) === auth.value;
  }

  public generateJWT(auth: any, options: any = {}): string {
    const newOptions = {
      // 7d, in miliseconds
      expiresIn: 60 * 60 * 24 * 7,
      ...(options || {})
    };
    const jwtSecret = this.configService.get('JWT_SECRET');

    return jwt.sign(
      {
        authId: auth._id,
        profileId: auth.profileId
      },
      jwtSecret,
      {
        expiresIn: newOptions.expiresIn
      }
    );
  }

  public verifyJWT(token: any) {
    try {
      const jwtSecret = this.configService.get('JWT_SECRET');
      return jwt.verify(token, jwtSecret);
    } catch (e) {
      return false;
    }
  }

  public async loginWithPassword(payload: LoginPasswordPayload) {
    // find profile by username, password and allow to login
    const profile = isEmail(payload.username)
      ? await this.profileService.findByEmail(payload.username)
      : await this.profileService.findByUsername(payload.username);
    if (!profile) {
      throw new HttpException({
        message: 'Username or email cannot be found!'
      }, 400);
    }
    // check password
    const authPw = await this.AuthModel.findOne({
      profileId: profile._id,
      type: 'password'
    });
    if (!authPw || !this.verifyPassword(payload.password, authPw)) {
      throw new HttpException({
        message: 'Password is incorrect!'
      }, 400);
    }

    if (profile.status !== 'active') {
      throw new HttpException({
        message: 'This account is not activated yet!'
      }, 400);
    }

    // generate jwt, should get remember me option
    const token = await this.generateJWT(authPw);
    return {
      token,
      user: plainToInstance(ProfileDto, profile)
    };
  }

  public async getProfileFromJwt(token: string) {
    const decodded = this.verifyJWT(token);
    if (!decodded) {
      throw new HttpException({
        message: 'Invalid token or session is expired'
      }, 400);
    }
    const { profileId } = decodded;
    const profile = await this.profileService.findById(profileId);
    // TODO - check active or something like that?
    return profile;
  }

  public async forgot(email: string) {
    const profile = await this.profileService.findByEmail(email);
    if (!profile) {
      // do not return error but success
      throw new EntityNotFoundException();
    }

    const token = randomString(15);
    const forgotLink = new URL(this.configService.get('FORGOT_PASSWORD_PAGE_URL'));
    forgotLink.searchParams.append('token', token);

    let forgot = await this.ForgotPasswordModel.findOne({
      profileId: profile._id
    });
    if (!forgot) {
      forgot = new this.ForgotPasswordModel();
    }
    forgot.createdAt = new Date();
    forgot.token = token;
    forgot.profileId = profile._id;
    await forgot.save();
    return true;
  }
}
