import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { EVENT, STANDARD_AD_CHANNEL } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, createAlias, isObjectId
} from 'src/core';
import { AdDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { AdCreatePayload, AdUpdatePayload } from 'src/payloads';
import { Ad, AdDocument } from 'src/schemas';

@Injectable()
export class AdService {
  constructor(
    @InjectModel(Ad.name) private readonly AdModel: Model<AdDocument>,
    private readonly fileService: FileService,
    private readonly queueMessageService: QueueMessageService
  ) { }

  public async generateSlug(title: string, id?: string | ObjectId) {
    // consider if need unique slug with post type
    const slug = createAlias(title);
    const query = { slug } as any;
    if (id) {
      query._id = { $ne: id };
    }
    const count = await this.AdModel.countDocuments(query);
    if (!count) {
      return slug;
    }

    return this.generateSlug(`${slug}1`, id);
  }

  public async create(payload: AdCreatePayload): Promise<AdDto> {
    const ad = await this.AdModel.create({
      ...payload
    });
    await this.queueMessageService.publish(STANDARD_AD_CHANNEL, {
      eventName: EVENT.CREATED,
      data: plainToInstance(AdDto, ad.toObject())
    });

    return plainToInstance(AdDto, ad.toObject());
  }

  public async findByIdOrSlug(id: string | ObjectId): Promise<AdDocument> {
    const query = id instanceof ObjectId || isObjectId(id)
      ? { _id: id }
      : { slug: id };
    return this.AdModel.findOne(query);
  }

  public async update(id: string | ObjectId, payload: AdUpdatePayload): Promise<AdDto> {
    const ad = await this.AdModel.findById(id);
    if (!ad) throw new EntityNotFoundException();
    merge(ad, payload);
    ad.updatedAt = new Date();
    await ad.save();
    return plainToInstance(AdDto, ad.toObject());
  }

  public async delete(id: string | ObjectId): Promise<any> {
    const ad = await this.AdModel.findById(id);
    if (!ad) throw new EntityNotFoundException();

    await ad.deleteOne();
    await this.queueMessageService.publish(STANDARD_AD_CHANNEL, {
      eventName: EVENT.DELETED,
      data: plainToInstance(AdDto, ad.toObject())
    });
    return true;
  }

  public async getDetails(id: string | ObjectId) {
    const ad = await this.findByIdOrSlug(id);
    if (!ad) throw new EntityNotFoundException();

    const dto = plainToInstance(AdDto, ad.toObject());
    if (ad.fileId) {
      const image = await this.getFileUploadedInfo(ad.fileId.toString());
      if (image.url) dto.setFileUrl(image);
    }
    return dto;
  }

  public async signUploadFileUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'ad-file',
      mediaType: payload?.mediaType || 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string, authenticated = false, expiresIn = 3600) {
    const file = await this.fileService.getPubicInfo({
      fileId,
      authenticated,
      expiresIn
    });
    if (!file) {
      throw new EntityNotFoundException();
    }
    return file;
  }
}
