import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { PageableData, populateDBSort } from 'src/core';
import { AdDto } from 'src/dtos';
import { FileService } from 'src/file-module/services';
import { AdSearchPayload } from 'src/payloads';
import { Ad, AdDocument } from 'src/schemas';

@Injectable()
export class AdSearchService {
  constructor(
    @InjectModel(Ad.name) private readonly AdModel: Model<AdDocument>,
    private readonly fileService: FileService
  ) { }

  public async populateData(data: AdDocument[]): Promise<AdDto[]> {
    const fileIds = data.filter((c) => !!c.fileId).map((c) => c.fileId.toString());

    const [files, fileThumbnails] = await Promise.all([
      fileIds.length > 0 ? this.fileService.getFilesPublicInfo({ fileIds, authenticated: true }) : {},
      fileIds.length > 0 ? this.fileService.getListFilesThumbnail({
        fileIds,
        authenticated: true,
        expiresIn: 3600
      }) : {}
    ]);

    return data.map((c) => {
      const ad = plainToInstance(AdDto, c);
      const file = c.fileId && files[c.fileId.toString()] ? files[c.fileId.toString()] : null;
      const thumbnail = c.fileId && fileThumbnails[c.fileId.toString()] ? fileThumbnails[c.fileId.toString()] : null;
      if (file) {
        ad.setFileUrl(file);
      }
      if (thumbnail) {
        ad.setThumbnailUrl(thumbnail);
      }
      return ad;
    });
  }

  public async advancedSearch(payload: AdSearchPayload): Promise<PageableData<AdDto>> {
    const query: FilterQuery<AdDocument> = {};

    if (payload.q) {
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          name: { $regex: regexp }
        }
      ];
    }

    if (payload.status) {
      query.status = payload.status;
    }

    if (payload.position) {
      query.position = payload.position;
    }

    if (payload.type) {
      query.type = payload.type;
    }

    const sort = populateDBSort(payload);

    const [data, total] = await Promise.all([
      this.AdModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.AdModel.countDocuments(query)
    ]);

    return {
      data: await this.populateData(data),
      total
    };
  }

  public async search(payload: AdSearchPayload): Promise<PageableData<AdDto>> {
    const query: FilterQuery<AdDocument> = {
      status: 'active'
    };

    if (payload.position) {
      query.position = payload.position;
    }

    if (payload.type) {
      query.type = payload.type;
    }

    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        }
      ];
    }

    const [data, total] = await Promise.all([
      this.AdModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.AdModel.countDocuments(query)
    ]);

    return {
      data: await this.populateData(data),
      total
    };
  }
}
