import {
  Prop, Schema, SchemaFactory, raw
} from '@nestjs/mongoose';
import { ObjectId } from 'mongodb';
import { HydratedDocument, Types, Schema as MongooseSchema } from 'mongoose';

@Schema({
  collection: 'videos'
})
export class Video {
  @Prop({
    trim: true,
    index: true
  })
  title: string;

  @Prop()
  description: string;

  @Prop({
    trim: true,
    index: true
  })
  slug: string;

  @Prop()
  seoTitle: string;

  @Prop()
  seoKeywords: string;

  @Prop()
  seoDesc: string;

  @Prop({
    default: 'active'
  })
  status: string;

  @Prop({
    index: true
  })
  fileId: string; // main file ID, use string format

  @Prop({
    default: 'in-queue',
    index: true
  })
  convertStatus: string;

  @Prop({
    default: 0
  })
  duration: number;

  @Prop({
    default: 'internal'
  })
  source: string;

  @Prop({
    default: null
  })
  externalId: string; // The external ID of the video is imported from different sources via csv file

  @Prop({
    // TODO - define schema separately
    // type: [string]
    type: Types.Map
  })
  thumbnails: string[];

  @Prop({
    default: ''
  })
  embeddedCode: string;

  @Prop({
    default: ''
  })
  teaser: string;

  @Prop({
    default: ''
  })
  url: string;

  @Prop({
    type: [{
      type: MongooseSchema.Types.ObjectId
    }],
    index: true
  })
  categoryIds: ObjectId[];

  @Prop({
    default: false,
    index: true
  })
  featured: boolean;


  @Prop({
    default: 0
  })
  price: number;

  @Prop({
    type: [{
      type: MongooseSchema.Types.ObjectId
    }],
    index: true
  })
  performerIds: ObjectId[];

  @Prop({
    type: [{
      type: String,
      lowercase: true
    }],
    index: true
  })
  tags: string[];

  @Prop(raw({
    likes: { type: Number, default: 0 },
    views: { type: Number, default: 0 },
    favourites: { type: Number, default: 0 },
    dislikes: { type: Number, default: 0 },
    shares: { type: Number, default: 0 }
  }))
  stats: {
    likes: number;
    views: number;
    favourites: number;
    dislikes: number;
    shares: number;
  };

  @Prop({
    default: false
  })
  isHD: boolean;

  @Prop({
    default: false
  })
  verified: boolean;

  @Prop({
    default: false
  })
  isByMember: boolean;

  @Prop({
    default: false
  })
  isDownloadedPoster: boolean;

  @Prop({
    index: true,
    type: MongooseSchema.Types.ObjectId
  })
  createdBy: ObjectId;

  @Prop({
    type: Date,
    default: Date.now
  })
  updatedAt: Date;

  @Prop({
    type: Date,
    default: Date.now
  })
  createdAt: Date;

  @Prop({
    type: [{
      type: MongooseSchema.Types.ObjectId
    }]
  })
  tagIds: ObjectId[];
}

export type VideoDocument = HydratedDocument<Video>;

export const VideoSchema = SchemaFactory.createForClass(Video);

// TODO - check if need unique index for this one?
VideoSchema.index({ source: 1, externalId: 1 }, { sparse: true });
VideoSchema.index({ slug: 1 }, { unique: true, sparse: true });
VideoSchema.index({ title: 'text' });
