import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import {
  EVENT,
  STATS_VIDEO_FIELDS,
  STAT_TYPES,
  VIDEO_CHANNEL,
  VIDEO_DOWNLOAD_POSTER_CHANNEL
} from 'src/constants';
import { QueueEvent, QueueMessageService, toObjectId } from 'src/core';
import { VideoDto } from 'src/dtos';
import {
  Reaction, ReactionDocument, Video, VideoDocument
} from 'src/schemas';
import {
  CategoryService, PerformerService, StatsService
} from 'src/services';

const TOPIC_HANDLE_COUNT_VIDEO = 'TOPIC_HANDLE_COUNT_VIDEO';
const TOPIC_HANDLE_UPDATE_VIDEO_POSTER = 'TOPIC_HANDLE_UPDATE_VIDEO_POSTER';

@Injectable()
export class VideoListener implements OnModuleInit {
  private logger = new Logger(VideoListener.name);

  constructor(
    private readonly queueMessageService: QueueMessageService,
    @InjectModel(Video.name) private readonly VideoModel: Model<VideoDocument>,
    private readonly categoryService: CategoryService,
    private readonly peformerService: PerformerService,
    private readonly statService: StatsService,
    @InjectModel(Reaction.name) private readonly ReactionModel: Model<ReactionDocument>
  ) { }

  onModuleInit() {
    this.listenQueueMessage();
  }

  async listenQueueMessage() {
    await this.queueMessageService.subscribe(VIDEO_CHANNEL, TOPIC_HANDLE_COUNT_VIDEO, this.handleUpdateRalatedData.bind(this));
    await this.queueMessageService.subscribe(VIDEO_DOWNLOAD_POSTER_CHANNEL, TOPIC_HANDLE_UPDATE_VIDEO_POSTER, this.handleUpdateVideoPoster.bind(this));
  }

  private async countCategoryVideos(categoryIds: string[] | ObjectId[]) {
    const data = await this.VideoModel.aggregate([
      {
        $match: {
          categoryIds: {
            $elemMatch: { $in: categoryIds.map((id) => toObjectId(id)) }
          }
        }
      },
      {
        $unwind: '$categoryIds'
      },
      {
        $group: {
          _id: '$categoryIds',
          count: {
            $sum: {
              $cond: [{ $eq: ['$status', 'active'] }, 1, 0]
            }
          }
        }
      }
    ]).allowDiskUse(true);

    await categoryIds.reduce(async (lp, catId) => {
      await lp;
      const found = data.length > 0 ? data.find((item) => `${item._id}` === `${catId}`) : null;
      if (found) {
        await this.categoryService.handleUpdateVideoStats(found._id, found?.count || 0);
      } else {
        await this.categoryService.handleUpdateVideoStats(catId, 0);
      }
      return Promise.resolve();
    }, Promise.resolve());
  }

  private async countPerformerVideos(performerIds: string[] | ObjectId[]) {
    const data = await this.VideoModel.aggregate([
      {
        $match: {
          performerIds: {
            $elemMatch: { $in: performerIds.map((id) => toObjectId(id)) }
          }
        }
      },
      {
        $unwind: '$performerIds'
      },
      {
        $group: {
          _id: '$performerIds',
          count: {
            $sum: {
              $cond: [{ $eq: ['$status', 'active'] }, 1, 0]
            }
          }
        }
      }
    ]).allowDiskUse(true);

    await performerIds.reduce(async (lp, performerId) => {
      await lp;
      const found = data.length > 0 ? data.find((item) => `${item._id}` === `${performerId}`) : null;
      if (found) {
        await this.peformerService.updateStats(found._id, { 'stats.videos': found.count || 0 });
      } else {
        await this.peformerService.updateStats(performerId, { 'stats.videos': 0 });
      }
      return Promise.resolve();
    }, Promise.resolve());
  }

  private async updateSystemStats(video: VideoDto) {
    const update = {};
    const totalVideo = await this.VideoModel.countDocuments({});
    update[`${STAT_TYPES.VIDEO}.${STATS_VIDEO_FIELDS.TOTAL}`] = totalVideo;
    update[`${STAT_TYPES.VIDEO}.${STATS_VIDEO_FIELDS.NEW}`] = totalVideo;

    const countUnverifiedVideo = await this.VideoModel.countDocuments({
      verified: false
    });
    update[`${STAT_TYPES.VIDEO}.${STATS_VIDEO_FIELDS.WAITING_VERIFIED}`] = countUnverifiedVideo;

    if (video.status === 'active') {
      const countActiveVideo = await this.VideoModel.countDocuments({
        status: 'active'
      });
      update[`${STAT_TYPES.VIDEO}.${STATS_VIDEO_FIELDS.ACTIVE}`] = countActiveVideo;
    }

    if (video.status === 'inactive') {
      const countInactiveVideo = await this.VideoModel.countDocuments({
        status: 'inactive'
      });
      update[`${STAT_TYPES.VIDEO}.${STATS_VIDEO_FIELDS.INACTIVE}`] = countInactiveVideo;
    }

    await this.statService.updateMultiStats(update);
  }

  private async handleUpdateRalatedData({ data }: QueueEvent<VideoDto>) {
    try {
      const {
        _id: videoId, categoryIds, performerIds
      } = data.data;
      await this.countCategoryVideos(categoryIds);
      await this.countPerformerVideos(performerIds);
      await this.updateSystemStats(data.data);
      const { eventName } = data;
      if ([EVENT.DELETED].includes(eventName)) {
        await this.ReactionModel.deleteMany({ objectId: videoId });
      }
    } catch (e) {
      this.logger.error(e);
    }
  }

  private async handleUpdateVideoPoster({ data }: QueueEvent<any>) {
    const { eventName } = data;
    if (eventName !== 'file_uploaded') {
      return;
    }
    const { postbackData, file, status } = data.data;

    const { videoId } = postbackData;
    const video = await this.VideoModel.findById(videoId);
    if (video && file) {
      video.set('fileId', file._id);
      video.set('status', status === 'error' ? 'inactive' : 'active');
      video.set('isDownloadedPoster', status === 'processed');
      await video.save();
    }
  }
}
