import { Injectable, OnModuleInit } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Job } from 'bullmq';
import { Model } from 'mongoose';

import { PARTNER_SOURCES, TUBECORPORATE_SOURCES, VIDEO_DOWNLOAD_POSTER_CHANNEL } from 'src/constants';
import { QueueService, isUrl } from 'src/core';
import { FileService } from 'src/file-module/services';
import { Video, VideoDocument } from 'src/schemas';

/**
 * schedule time, need this flag to check in case
 */
const SCHEDULE_JOB_REPEAT_PATTERN = '*/1 * * * *'; // run job every 10m

@Injectable()
export class DownloadPosterFormTC implements OnModuleInit {
  constructor(
    private readonly queueService: QueueService,
    @InjectModel(Video.name) private readonly VideoModel: Model<VideoDocument>,
    private readonly fileService: FileService
  ) { }

  onModuleInit() {
    this.defineJobs();
  }

  private async addJobQueue() {
    await this.queueService.add('download_poster_form_tc', 'download_poster_form_tc', {
      jobOptions: {
        repeat: {
          pattern: SCHEDULE_JOB_REPEAT_PATTERN
        },
        jobId: 'download_poster_form_tc',
        removeOnComplete: true,
        removeOnFail: true
      },
      jobUnique: true
    });
  }

  private async defineJobs() {
    await this.addJobQueue();

    this.queueService.processWorker('download_poster_form_tc', this.handleDownloadPosterFormTc.bind(this));
  }

  async handleDownloadPosterFormTc(job: Job) {
    // compare with the same pattern, because if we change repeat patter, there are 2 jobs will be created and executed.
    if (!this.queueService.isInActiveQueueJob(job)) return;
    const {
      repeat
    } = job.opts;
    if (repeat.jobId !== 'download_poster_form_tc' || repeat?.pattern !== SCHEDULE_JOB_REPEAT_PATTERN) return;

    const tcVideos = await this.VideoModel.find({
      source: {
        $in: [...TUBECORPORATE_SOURCES, ...PARTNER_SOURCES]
      },
      isDownloadedPoster: false
      // status: 'active'
    })
      .sort({
        createdAt: 1
      })
      .limit(10);

    await tcVideos.reduce(async (lp, video) => {
      await lp;
      if (video.thumbnails && video.thumbnails.length) {
        await this.fileService.downloadPoster({
          url: isUrl(video.thumbnails[0]) ? video.thumbnails[0] : `http:${video.thumbnails[0]}`,
          notification: {
            channel: VIDEO_DOWNLOAD_POSTER_CHANNEL,
            data: {
              videoId: video._id.toString()
            }
          }
        });
      }
      Promise.resolve();
    }, Promise.resolve());
  }
}
