import {
  ReadStream,
  cpSync,
  createWriteStream,
  existsSync,
  renameSync,
  unlinkSync,
  writeFileSync
} from 'fs';
import { join } from 'path';

import { Injectable } from '@nestjs/common';
import { mkdirp } from 'mkdirp';

import { randomString, toPosixPath } from 'src/core';

import { AbstractFileUploadService } from './abstract-file-upload.service';
import { PUBLIC_DIR } from '../constants';
import { IFileUpload, IFileUploadResponse } from '../interfaces';


@Injectable()
export class LocalFileUploadService extends AbstractFileUploadService {
  public async upload({
    localPath,
    body,
    toDir,
    fileName,
    fileExt,
    fileId,
    acl,
    fileType,
    deleteLocalFileAfterUploaded = true
  }: IFileUpload): Promise<IFileUploadResponse> {
    let localAbsolutePath;
    // process upload or write file
    if (!localPath && !body) throw new Error('Missing local path or buffer!');
    if (localPath) {
      if (!existsSync(localPath)) {
        const absolutePath = join(PUBLIC_DIR, localPath);
        if (!existsSync(absolutePath)) throw new Error('File not found!');

        localAbsolutePath = absolutePath;
      } else {
        localAbsolutePath = localPath;
      }
    }

    const name = [fileName || randomString(5), fileExt].filter((n) => !!n).join('.');
    const publicDir = toDir || PUBLIC_DIR;
    let newAbsoluteDir;
    switch (fileType) {
      case 'image':
        newAbsoluteDir = acl === 'public-read'
          ? join(publicDir, 'images', fileId.toString())
          : join(publicDir, 'images', 'protected', fileId.toString());
        break;
      case 'video':
        newAbsoluteDir = acl === 'public-read'
          ? join(publicDir, 'videos', fileId.toString())
          : join(publicDir, 'videos', 'protected', fileId.toString());
        break;
      default: break;
    }

    const newAbsolutePath = join(newAbsoluteDir, name);
    let newPath = toPosixPath(newAbsolutePath.replace(publicDir, ''));
    if (newPath.charAt(0) !== '/') newPath = `/${newPath}`;

    if (deleteLocalFileAfterUploaded && localAbsolutePath) {
      // move file instead
      mkdirp.sync(newAbsoluteDir);
      // check this option, if rename in diffent disk we need to copy and delete file instead
      try {
        renameSync(localAbsolutePath, newAbsolutePath);
      } catch {
        cpSync(localPath, newAbsolutePath);
        unlinkSync(localAbsolutePath);
      }
      return {
        absolutePath: newAbsolutePath,
        path: newPath
      };
    }
    if (body) {
      // create from buffer
      if (body instanceof ReadStream) {
        const ws = createWriteStream(newAbsolutePath);
        body.pipe(ws);
      } else {
        mkdirp.sync(newAbsoluteDir);
        writeFileSync(newAbsolutePath, body);
      }
    } else {
      // copy file
      cpSync(localPath, newAbsolutePath);
    }

    return {
      absolutePath: newAbsolutePath,
      path: newPath
    };
  }
}
