import { spawn } from 'child_process';
import { existsSync } from 'fs';
import { cpus } from 'os';
import { join } from 'path';

import { path as ffprobePath } from '@ffprobe-installer/ffprobe';
import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as ffprobe from 'node-ffprobe';

import {
  getFilePath, randomString, secondsToTime, toPosixPath
} from 'src/core';

import { PUBLIC_DIR } from '../constants';
import { ICreateVideoThumbnailResponse, ICreateVideoThumbnailsOptions } from '../interfaces';

export interface IVideoMetadata {
  index: number;
  /**
   * h264, hevc
   */
  codec_name: string;
  /**
   * 'H.264 / AVC / MPEG-4 AVC / MPEG-4 part 10',
   */
  codec_long_name: string;
  /**
   * main
   */
  profile: string;
  /**
   *  'video'
   */
  codec_type: string;
  /**
   * 'avc1'
   */
  codec_tag_string: string;
  codec_tag: string;
  width: number;
  height: number;
  coded_width: number;
  coded_height: number;
  closed_captions: number;
  film_grain: number;
  has_b_frames: number;
  /**
   * yuv420p
   */
  pix_fmt: string;
  level: number;
  color_range: string;
  color_space: string;
  color_transfer: string;
  color_primaries: string;
  chroma_location: string;
  field_order: string;
  refs: number;
  /**
   * true / false
   */
  is_avc: string;
  nal_length_size: string;
  id: string;
  r_frame_rate: string;
  avg_frame_rate: string;
  time_base: string;
  start_pts: number;
  /**
   * string number
   */
  start_time: string;
  duration_ts: number;
  /**
   * string number
   */
  duration: string;
  bit_rate: string;
  bits_per_raw_sample: string;
  nb_frames: string;
  extradata_size: number;
  disposition: any[];
  tags: any[];
}

export interface IConvertOptions {
  toPath?: string;
  size?: string; // https://github.com/fluent-ffmpeg/node-fluent-ffmpeg#video-frame-size-options
  watermark?: any;
}

export interface IConvertResponse {
  fileName: string;
  toPath: string;
}

@Injectable()
export class FileVideoService {
  private FFEMPEG_PATH = process.env.FFEMPEG_PATH || 'ffmpeg';

  constructor(
    private readonly configService: ConfigService
  ) { }

  private setupConfig() {
    const FFPROBE_PATH = this.configService.get('FFPROBE_PATH');
    ffprobe.FFPROBE_PATH = FFPROBE_PATH || ffprobePath;
  }

  private getAbsolutePath(filePath) {
    let absolutePath = filePath;
    if (!existsSync(filePath)) {
      absolutePath = join(PUBLIC_DIR, absolutePath);
      if (!existsSync(absolutePath)) throw new Error('File not found!');
    }

    return absolutePath;
  }

  private getThreadsLimit() {
    const cpuCount = cpus().length;
    const defaultNum = Math.ceil(cpuCount / 2);

    if (process.env.FFMPEG_CPU_LIMIT) {
      const num = parseInt(process.env.FFMPEG_CPU_LIMIT, 10);
      if (num > cpuCount) return defaultNum;
      if (num < 1) return 1;
    }
    return defaultNum;
  }

  /**
   * resize iamge file
   * @param filePath url to video file or path in local
   * @returns
   */
  public async getMetaData(filePath: string) {
    this.setupConfig();
    return ffprobe(filePath);
  }

  public async getVideoMetadata(filePath: string): Promise<IVideoMetadata> {
    this.setupConfig();
    const absolutePath = this.getAbsolutePath(filePath);

    const { streams } = await ffprobe(absolutePath);
    if (!streams?.length) return null;
    const video = streams.find((s) => s.codec_type === 'video');
    if (!video) return null;

    return video;
  }

  public async convert2Mp4(
    filePath: string,
    options = {} as IConvertOptions
  ): Promise<IConvertResponse> {
    const absolutePath = this.getAbsolutePath(filePath);
    const fileName = `${randomString(5)}}.mp4`;
    const toPath = options.toPath || join(getFilePath(absolutePath), fileName);
    const threads = this.getThreadsLimit();

    return new Promise((resolve, reject) => {
      let outputOptions = `-vcodec libx264 -pix_fmt yuv420p -profile:v baseline -level 3.0 -movflags +faststart -strict experimental -preset fast -threads ${threads} -crf 23`;
      if (options.size) {
        const sizes = options.size.split('x');
        const width = sizes[0];
        // retain aspect ratio just give height as -1 and it will automatically resize based on the width
        const height = sizes.length > 1 ? sizes[1] : '-1  ';
        outputOptions += ` -vf scale="${width}:${height}"`;
      }

      if (options.watermark) {
        const useImage = false;
        if (!useImage) {
          const {
            text,
            color = '#ffffff',
            fontSize = 24,
            opacity = 1,
            bottom = 10,
            top = 20,
            left = 10,
            align = 'top' // support top, middle, bottom only
          } = options.watermark;
          let textPos = 'x=(w-text_w)/2:y=(h-text_h)/2'; // middle
          // center x=(w-text_w)/2:y=(h-text_h)/2
          switch (align) {
            case 'bottom':
              textPos = `x=${left}:y=H-th-${bottom}`;
              break;
            case 'top':
              textPos = `x=${left}:y=${top}`;
              break;
            default:
              break;
          }

          const fontfile = join(__dirname, '..', '..', '..', '..', 'fonts', 'montserrat.ttf');
          outputOptions += ` -vf drawtext=text='${text}':${textPos}:fontsize=${fontSize}:fontfile=${fontfile}:fontcolor=${color}@${opacity}`;
        } else {
          const watermarkFile = null; // path to watermark image
          if (watermarkFile && existsSync(watermarkFile)) {
            // input of watermark must be first
            // center of video
            // outputOptions = `-i ${watermarkFile} -filter_complex "overlay=W-w-10:H-h-10" ${outputOptions}`;
            // [1]colorchannelmixer=aa=0.5,scale=iw*0.1:-1[wm] // opacity to 50%
            // bottom right
            outputOptions = `-i ${watermarkFile} -filter_complex "[1]scale=iw*0.1:-1[wm];[0][wm]overlay=x=(main_w-overlay_w):y=(main_h-overlay_h)" ${outputOptions}`;
          }
        }
      }
      const q = `${this.FFEMPEG_PATH} -i ${absolutePath} ${outputOptions} ${toPath}`;
      const command = spawn(q, [], {
        shell: true,
        stdio: 'ignore' // do not print stdout
      });
      command.on('exit', (code) => {
        if (!code) {
          resolve({
            fileName,
            toPath
          });
          return;
        }
        reject(new Error('Convert video error'));
      });
    });
  }

  /**
   * get video duration
   * @param filePath absolute path
   * @returns
   */
  public async getDuration(filePath: string): Promise<number> {
    const absolutePath = this.getAbsolutePath(filePath);
    const metadata = await this.getMetaData(absolutePath);
    if (!metadata) return null;
    return parseInt(metadata.format.duration, 10);
  }

  public async createThumbnails({
    filePath,
    timemarks = [],
    width,
    height,
    toDir
  }: ICreateVideoThumbnailsOptions): Promise<ICreateVideoThumbnailResponse[]> {
    const absolutePath = this.getAbsolutePath(filePath);

    // ffmpeg -ss 00:00:01.00 -i input.mp4 -vf 'scale=320:320:force_original_aspect_ratio=decrease' -vframes 1 output.jpg
    // ffmpeg -ss t -i source.mp4 -f mjpeg -vframe 1 -s wxh -
    let timeToGet = timemarks;
    if (!timemarks?.length) {
      const duration = await this.getDuration(absolutePath);
      if (!duration) timeToGet = ['00:00:01'];
      else {
        const s = secondsToTime(Math.ceil(duration / 2));
        timeToGet = [
          `${s.h}:${s.m}:${s.s}`
        ];
      }
    }
    let imgWidth = width || -1;
    const imgHeight = height || -1;
    if (imgWidth === -1 && imgHeight === -1) {
      imgWidth = 320;
    }
    const num = 1;
    return timeToGet.reduce(async (res, timemark) => {
      const results = await res;
      const fileName = `screenshot${num}.jpg`;
      const output = join(toDir, fileName);
      const q = `${this.FFEMPEG_PATH} -ss ${timemark} -i ${filePath} -vf "scale=${imgWidth}:${imgHeight}:force_original_aspect_ratio=decrease" -vframes 1 ${output}`;
      const r = await new Promise((resolve, reject) => {
        const command = spawn(q, [], {
          shell: true,
          stdio: 'ignore' // do not print stdout
        });
        command.on('exit', (code) => {
          if (!code) {
            resolve({
              fileName,
              path: toPosixPath(output)
            });
            return;
          }
          reject(new Error('Convert video error'));
        });
      });
      results.push(r);
      return results;
    }, Promise.resolve([]));
  }

  /**
   * check if this video support html5, we don't need to convert to h264 if any?
   * @param filePath
   * @returns
   */
  public async isSupportHtml5(filePath: string) {
    const meta = await this.getMetaData(filePath);
    if (!meta?.streams?.length) return false;
    const videoStream = meta.streams.find((s) => s.codec_type === 'video');

    // TODO - check if pix_fmt: 'yuv420p'
    return ['h264', 'vp8'].includes(videoStream.codec_name) && ['mp4', 'webm'].includes(meta.format?.format_name);
  }
}
