import { existsSync, readFileSync, unlinkSync } from 'fs';
import { basename, join } from 'path';

import { HttpService } from '@nestjs/axios';
import {
  HttpException, Injectable, OnModuleInit
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { InjectModel } from '@nestjs/mongoose';
import axios from 'axios';
import { plainToInstance } from 'class-transformer';
import * as moment from 'moment';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';
import { firstValueFrom } from 'rxjs';

import {
  EntityNotFoundException,
  QueueEvent,
  QueueMessageService,
  generateUuid,
  getExt,
  isObjectId,
  isUrl,
  randomString,
  toPosixPath
} from 'src/core';


import { AbstractFileUploadService } from './abstract-file-upload.service';
import { FileImageService } from './file-image.service';
import { FileVideoService, IConvertOptions, IConvertResponse } from './file-video.service';
import { LocalFileUploadService } from './local-file-upload.service';
import {
  FILE_MEDIA_CONVERT_CHANNEL,
  FILE_UPLOADED_EVENT,
  HANDLE_MEDIA_CONVERT_TOPIC,
  MAX_UPLOAD_SIZE_IMAGE_IN_MB,
  MAX_UPLOAD_SIZE_VIDEO_IN_MB,
  FILE_STATUS,
  THUMBNAIL_WIDTH,
  PUBLIC_DIR
} from '../constants';
import { FileDto } from '../dtos';
import { IMulterFileUpload, IPublicFileInfo, PresignedUploadUrl } from '../interfaces';
import {
  DownloadPosterPayload, GetFilePublicPayload, GetFilesPublicPayload, SignedUploadRequestPayload
} from '../payloads';
import { File } from '../schemas';

@Injectable()
export class FileService implements OnModuleInit {
  constructor(
    @InjectModel(File.name) private readonly FileModel: Model<File>,
    private readonly imageService: FileImageService,
    private readonly videoService: FileVideoService,
    private readonly localFileUploadService: LocalFileUploadService,
    private readonly queueMessageService: QueueMessageService,
    private readonly configService: ConfigService,
    private readonly httpService: HttpService
  ) { }

  onModuleInit() {
    this.handleListener();
  }

  private async handleListener() {
    await this.queueMessageService.subscribe(FILE_MEDIA_CONVERT_CHANNEL, HANDLE_MEDIA_CONVERT_TOPIC, this.processMediaConvertQueue.bind(this));
  }

  private getFileUploadService(name = 'local'): AbstractFileUploadService {
    // TODO - implement other method
    switch (name) {
      case 'local':
        return this.localFileUploadService;
      default: throw new HttpException('Not available', 400);
    }
  }

  private async notifyFileProcessed(file: File | any, status = 'success') {
    const expiresIn = 3600;
    const authenticated = true;
    if (file.notification?.channel) {
      // check webhook or notification channel via queue
      const updated = await this.FileModel.findOne({ _id: file._id });
      const dto = plainToInstance(FileDto, updated.toObject());
      await this.queueMessageService.publish(file.notification.channel, {
        eventName: FILE_UPLOADED_EVENT,
        data: {
          postbackData: file.notification.data || {},
          file: await dto.getPostbackInfo(authenticated, expiresIn),
          status
        }
      });
    } else if (file.notification?.webhookUrl) {
      // check webhook or notification channel via queue
      const updated = await this.FileModel.findOne({ _id: file._id });
      const dto = plainToInstance(FileDto, updated.toObject());
      // TODO - log error if have
      await firstValueFrom(this.httpService.post(file.notification.webhookUrl, {
        postbackData: file.notification.data || {},
        file: await dto.getPostbackInfo(authenticated, expiresIn),
        status
      }));
    }
  }

  private async notifyFileDeleted(file: File, status = 'deleted') {
    if (file.notification?.channel) {
      const dto = plainToInstance(FileDto, file);
      await this.queueMessageService.publish(file.notification.channel, {
        eventName: FILE_UPLOADED_EVENT,
        data: {
          postbackData: file.notification.data || {},
          file: await dto.getPostbackInfo(),
          status
        }
      });
    } else if (file.notification.webhookUrl) {
      // check webhook or notification channel via queue
      const dto = plainToInstance(FileDto, file);
      // TODO - log error if have
      await firstValueFrom(this.httpService.post(file.notification.webhookUrl, {
        postbackData: file.notification.data || {},
        file: await dto.getPostbackInfo(),
        status
      }));
    }
  }

  public async addQueueMediaConvert(file: File | any) {
    const dto = plainToInstance(FileDto, file);
    await this.queueMessageService.publish(FILE_MEDIA_CONVERT_CHANNEL, {
      eventName: 'created',
      data: dto
    });
  }

  private async processMediaConvertQueue({ data }: QueueEvent<FileDto>) {
    const { eventName } = data;
    if (eventName !== 'created') return;

    const dto: FileDto = data.data;
    switch (dto.mediaType) {
      case 'image':
        await this.processImageLocalUpload(dto._id, {
          path: dto.path
          // TODO - define more info
        } as any);
        break;
      case 'video':
        await this.processVideoLocalUpload(dto._id, {
          path: dto.path,
          mime: dto.mime
        } as any);
        break;
      default: break;
    }
  }

  /**
   * generate file for direct upload
   * @param payload
   * @returns
   */
  public async getPresignedUploadUrl(payload: SignedUploadRequestPayload): Promise<PresignedUploadUrl> {
    const maxUploadSizeInMB = payload.mediaType === 'video' ? MAX_UPLOAD_SIZE_VIDEO_IN_MB : MAX_UPLOAD_SIZE_IMAGE_IN_MB;
    const ext = payload.filename ? getExt(payload.filename) : '';
    let startWith = '';
    switch (payload.mediaType) {
      case 'video':
        startWith = 'videos/original/';
        break;
      case 'csv':
        startWith = 'csv/original/';
        break;
      default:
        startWith = 'images/original/';
        break;
    }
    const filename = `${startWith}${generateUuid()}${ext}`;
    const file = await this.FileModel.create({
      ...payload,
      uploadedBy: payload.uploaderId,
      status: FILE_STATUS.PENDING_UPLOAD,
      source: 'local',
      path: filename,
      createdAt: new Date(),
      updatedAt: new Date()
    });
    await file.save();

    // process image upload
    // TODO - define other type
    const uploadUrl = this.configService.get('FILE_UPLOAD_BASE_URL').replace('{fileId}', file._id);
    return {
      uploadUrl,
      bodyBinary: false,
      fields: [],
      fileId: file._id,
      status: FILE_STATUS.PENDING_UPLOAD,
      maxUploadSizeInMB
    };
  }

  public async findById(id: string | ObjectId) {
    const file = await this.FileModel.findById(id);
    if (!file) return null;
    // if (file?.status === FILE_STATUS.UPLOADED) return null;
    const dto = plainToInstance(FileDto, file.toObject());
    return dto;
  }

  public async deletePhysicalFiles(id: string | ObjectId | File) {
    const file = ((id instanceof File || (id as any)._id) ? id : await this.FileModel.findById(id)) as File;
    if (!file) return true;

    const filePaths = [
      file.path,
      file.blurImagePath,
      file.thumbnailPath
    ].filter((p) => !!p);
    if (file.poster?.key) {
      filePaths.push(file.poster.key);
    }
    if (file.videos?.length) {
      filePaths.push(
        ...file.videos.map((v) => v.key)
      );
    }
    // try to lookup local server, in case file error or some reasons like that
    const uploadDir = this.configService.get('PUBLIC_DIR');
    await filePaths.reduce(async (lp, filePath) => {
      await lp;
      if (existsSync(filePath)) unlinkSync(filePath);
      else if (existsSync(join(uploadDir, filePath))) unlinkSync(join(uploadDir, filePath));

      return Promise.resolve();
    }, Promise.resolve());

    // delete S3 if needed
    return true;
  }

  /**
   * delete single files
   * @param id
   * @returns
   */
  public async deleteFile(id: string | ObjectId | File) {
    const file = ((id instanceof Model || (id as any)._id) ? id : await this.FileModel.findById(id)) as any;
    if (!file) return true;
    await this.FileModel.deleteOne({ _id: file._id });
    await this.deletePhysicalFiles(file);
    await this.notifyFileDeleted(file);
    return true;
  }

  /**
   * delete multiple files
   * @param ids
   * @returns
   */
  public async deleteFiles(ids: (string | ObjectId | File)[]) {
    await ids.reduce(async (lp, id) => {
      await lp;
      await this.deleteFile(id);
      return Promise.resolve();
    }, Promise.resolve());
    return true;
  }

  // eslint-disable-next-line
  public async getSignedUrl(fileId: string, bucket = null, expiresIn = 15 * 60) {
    if (isObjectId(fileId)) {
      // find file and signed URL
      const file = await this.FileModel.findById(fileId);
      if (!file) return null;
      // TODO - implement me
      return '';
    }

    // if URL is full path, just need path name
    const key = isUrl(fileId) ? new URL(fileId).pathname : fileId;
    // TODO - implement me
    return key;
  }

  /**
   * process convert, create thumbnail, etc... for the uploaded image
   * @param fileId
   * @param uploadedFile
   * @returns
   */
  public async processImageLocalUpload(fileId: string | ObjectId | Model<File> | any, uploadedFile: IMulterFileUpload) {
    const file: any = fileId instanceof Model ? fileId : await this.FileModel.findById(fileId);
    if (!file) throw new EntityNotFoundException();
    try {
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          format: uploadedFile.mimetype,
          status: FILE_STATUS.PROCESSING,
          updatedAt: new Date()
        }
      });
      // if mimetype is not image, ignore?
      if (!(uploadedFile.mimetype || '').includes('image')) {
        throw new HttpException('Invalid image!', 400);
      }
      // for now we will create a small thumbnail and down quality for original image only
      // TODO - add to Queue
      const metadata = await this.imageService.getMetaData(uploadedFile.path);
      // if it is not in the supported list, skip convert?
      const isSupportedFormat = this.imageService.isSupportedFormat(metadata.format);
      // allow max 2k for width
      let width = metadata.width > 2000 ? 2000 : metadata.width;
      let height; // Math.ceil((metadata.height * width) / 2000);
      // overwrite default config if image height is too high
      if (metadata.height > 4000) {
        height = 2000;
        width = undefined;
      }
      if (file.params?.forceImageWidth) {
        width = file.params.forceImageWidth;
        height = file.params.forceImageHeight || undefined;
      } else if (file.params?.forceImageHeight) {
        height = file.params.forceImageHeight;
        width = file.params.forceImageWidth || undefined;
      }
      const newPathBuffer = !isSupportedFormat
        ? readFileSync(uploadedFile.path)
        : await this.imageService.resize(uploadedFile.path, {
          width,
          height,
          // overwrite default quality, should check this option if needed
          quality: 80,
          format: metadata.format
        }) as Buffer;
      const fileUpload = this.getFileUploadService('local'); // support local only for now
      const res = await fileUpload.upload({
        body: newPathBuffer,
        fileType: 'image',
        acl: file.acl,
        fileId: file._id,
        fileExt: metadata?.format || 'png',
        deleteLocalFileAfterUploaded: true
      });
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          path: toPosixPath(res.path),
          width,
          height,
          updatedAt: new Date()
        }
      });

      // create thumbnails and blur image then upload?
      if (isSupportedFormat && file.params?.createThumbnail !== false) {
        const thumbnailThumbBuffer = await this.imageService.resize(newPathBuffer, {
          width: THUMBNAIL_WIDTH,
          quality: 60,
          format: metadata.format
        }) as Buffer;

        const thumbUploadRes = await fileUpload.upload({
          body: thumbnailThumbBuffer,
          fileType: 'image',
          acl: 'public-read',
          fileId: file._id,
          fileName: `${randomString(5)}_thumb.${metadata?.format || 'png'}`
        });
        await this.FileModel.updateOne({ _id: file._id }, {
          $set: {
            thumbnailPath: toPosixPath(thumbUploadRes.path),
            updatedAt: new Date()
          }
        });
      } else {
        await this.FileModel.updateOne({ _id: file._id }, {
          $set: {
            thumbnailPath: toPosixPath(res.path),
            updatedAt: new Date()
          }
        });
      }

      let blurImagePath;
      if (isSupportedFormat && file.params?.createBlurImage !== false) {
        const blurBuffer = await this.imageService.blur(newPathBuffer, {
          sigma: 6, // a value between 0.3 and 1000 representing the sigma of the Gaussian mask, where sigma = 1 + radius / 2
          width: 100,
          quality: 30
        }) as Buffer;
        const blurUploadRes = await fileUpload.upload({
          body: blurBuffer,
          fileType: 'image',
          acl: 'public-read',
          fileId: file._id,
          fileName: `${randomString(5)}_bl.jpg`
        });
        blurImagePath = blurUploadRes.path;
      }

      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          blurImagePath: toPosixPath(blurImagePath),
          updatedAt: new Date(),
          status: FILE_STATUS.PROCESSED
        }
      });

      if (existsSync(uploadedFile.path)) unlinkSync(uploadedFile.path);

      await this.notifyFileProcessed(file, 'processed');

      return true;
    } catch (e) {
      const error = await e;
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          status: FILE_STATUS.ERROR,
          error: error.stack,
          updatedAt: new Date()
        }
      });

      await this.deletePhysicalFiles(file);
      await this.notifyFileProcessed(file, 'error');
      // remove original file as well
      if (existsSync(uploadedFile.path)) unlinkSync(uploadedFile.path);
      throw error;
    }
  }

  public async processVideoLocalUpload(fileId: string | ObjectId | Model<File> | any, uploadedFile: IMulterFileUpload) {
    // convert to multiple file size and upload
    const file: any = fileId instanceof Model ? fileId : await this.FileModel.findById(fileId);
    if (!file) throw new EntityNotFoundException();
    const filesShouldBeDeletedOnCrash = [uploadedFile.path];
    try {
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          format: uploadedFile.mimetype,
          status: FILE_STATUS.PROCESSING,
          updatedAt: new Date()
        }
      });
      // if mimetype is not image, ignore?
      // TODO - validate other file format
      // if (!(uploadedFile.mimetype || '').includes('video')) {
      //   throw new HttpException('Invalid video!', 400);
      // }
      // for now we will create a small thumbnail and down quality for original image only
      let metadata = await this.videoService.getVideoMetadata(uploadedFile.path);
      const isSupportHtml5 = await this.videoService.isSupportHtml5(uploadedFile.path);

      let mp4Response: IConvertResponse;
      if (!isSupportHtml5) {
        // check for 2k, 4k here
        // with mp4 h264, it is max 4k video only
        // TODO - check max file resolution support
        const options = {} as IConvertOptions;
        if (metadata.width > 3840) options.size = '3840x-1';
        mp4Response = await this.videoService.convert2Mp4(uploadedFile.path, options);
        metadata = await this.videoService.getVideoMetadata(mp4Response.toPath);
      } else {
        // TODO - check if need to move uploaded file to new path?
        mp4Response = {
          toPath: uploadedFile.path,
          fileName: basename(uploadedFile.path)
        };
      }

      const fileUpload = this.getFileUploadService('local'); // support local only for now
      const res = await fileUpload.upload({
        localPath: mp4Response.toPath,
        fileType: 'video',
        acl: file.acl,
        fileId: file._id,
        fileExt: mp4Response.toPath.split('.').pop(),
        deleteLocalFileAfterUploaded: true
      });
      const createThumbFrom = res.absolutePath;
      // TODO - check if need to create multi presets
      const videos = [{
        key: toPosixPath(res.path),
        url: toPosixPath(res.path),
        acl: file.acl,
        width: metadata.width,
        height: metadata.height,
        format: 'mp4'
      }];
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          path: toPosixPath(res.path),
          videos,
          width: metadata.width,
          height: metadata.height,
          meta: metadata,
          updatedAt: new Date()
        }
      });

      // create thumbnails and blur image then upload?
      let thumbnailPath;
      if (file.params?.createThumbnail !== false) {
        const thumbnails = await this.videoService.createThumbnails({
          filePath: createThumbFrom,
          toDir: PUBLIC_DIR,
          width: 480
        });
        thumbnailPath = thumbnails[0]?.path;
        filesShouldBeDeletedOnCrash.push(...(thumbnails).map((t) => t.path));

        const thumbUploadRes = await fileUpload.upload({
          localPath: thumbnailPath,
          fileType: 'video',
          acl: 'public-read',
          fileId: file._id,
          fileName: `${randomString(5)}_thumb.jpg`,
          deleteLocalFileAfterUploaded: false
        });
        await this.FileModel.updateOne({ _id: file._id }, {
          $set: {
            thumbnailPath: toPosixPath(thumbUploadRes.path),
            updatedAt: new Date()
          }
        });
      }

      if (thumbnailPath && file.params?.createBlurImage !== false) {
        const blurBuffer = await this.imageService.blur(thumbnailPath, {
          sigma: 6, // a value between 0.3 and 1000 representing the sigma of the Gaussian mask, where sigma = 1 + radius / 2
          width: 100,
          quality: 30
        }) as Buffer;
        const blurUploadRes = await fileUpload.upload({
          body: blurBuffer,
          fileType: 'video',
          acl: 'public-read',
          fileId: file._id,
          fileName: `${randomString(5)}_bl.jpg`
        });
        await this.FileModel.updateOne({ _id: file._id }, {
          $set: {
            blurImagePath: toPosixPath(blurUploadRes.path),
            updatedAt: new Date(),
            status: FILE_STATUS.PROCESSED
          }
        });
      }

      // delete thumbnail file if have
      if (existsSync(thumbnailPath)) unlinkSync(thumbnailPath);
      // remove original file
      if (existsSync(uploadedFile.path)) unlinkSync(uploadedFile.path);

      await this.notifyFileProcessed(file, 'processed');

      return true;
    } catch (e) {
      const error = await e;
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          status: FILE_STATUS.ERROR,
          error: error.stack,
          updatedAt: new Date()
        }
      });

      await this.deletePhysicalFiles(file);
      await this.notifyFileProcessed(file, 'error');
      filesShouldBeDeletedOnCrash.forEach((p) => {
        if (existsSync(p)) unlinkSync(p);
      });

      throw error;
    }
  }

  public async processFileUpload(fileId: string | ObjectId | Model<File> | any, uploadedFile: IMulterFileUpload) {
    const file: any = fileId instanceof Model ? fileId : await this.FileModel.findById(fileId);
    if (file?.status !== FILE_STATUS.PENDING_UPLOAD) {
      throw new HttpException('Invalid file status', 400);
    }
    if (moment(file.createdAt).add(4, 'hours').isBefore(new Date())) {
      throw new HttpException('Expired!', 400);
    }
    if (file.mediaType === 'image') return this.processImageLocalUpload(file, uploadedFile);

    // TODO - add queue
    if (file.mediaType === 'video') {
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          status: 'in_queue',
          path: uploadedFile.path,
          size: uploadedFile.size,
          mime: uploadedFile.mimetype
        }
      });

      // add queue process
      const newFile = await this.FileModel.findById(file._id);
      return this.queueMessageService.publish(FILE_MEDIA_CONVERT_CHANNEL, {
        eventName: 'created',
        data: plainToInstance(FileDto, newFile.toObject())
      });
    }
    if (file.mediaType === 'csv') {
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          status: FILE_STATUS.UPLOADED,
          path: uploadedFile.path,
          size: uploadedFile.size,
          mime: uploadedFile.mimetype
        }
      });
      // add queue process
      const newFile = await this.FileModel.findById(file._id);
      await this.notifyFileProcessed(newFile, 'processed');
    }
    return null;
  }

  /**
   * get public info of single file
   * @param payload
   * @returns
   */
  public async getPubicInfo(payload: GetFilePublicPayload) {
    const file = await this.FileModel.findById(payload.fileId);
    if (!file) throw new EntityNotFoundException();

    const dto = plainToInstance(FileDto, file.toObject());
    return dto.getPublicInfo(payload.authenticated, payload.expiresIn, true);
  }

  /**
   * get public info of the list
   */
  public async getFilesPublicInfo(payload: GetFilesPublicPayload) {
    const files = await this.FileModel.find({
      _id: {
        $in: payload.fileIds
      }
    });
    if (!files.length) return [];
    return files.reduce(async (lp, file) => {
      const results = await lp;
      const dto = plainToInstance(FileDto, file.toObject());
      const response = await dto.getPublicInfo(payload.authenticated, payload.expiresIn);
      results[file._id.toString()] = response;
      return results;
    }, {} as any) as Record<string, IPublicFileInfo>;
  }

  /**
   * get thumbnail url and blurred image of a file
   * @param fileId
   */
  public async getThumbnail(payload: GetFilePublicPayload) {
    const file = await this.FileModel.findById(payload.fileId);
    if (!file) throw new EntityNotFoundException();

    const dto = plainToInstance(FileDto, file.toObject());
    return dto.getThumbnail(payload.authenticated, payload.expiresIn);
  }

  /**
   * get thumbnail url and blurred image url of list files
   * @param fileId
   */
  public async getListFilesThumbnail(payload: GetFilesPublicPayload) {
    const files = await this.FileModel.find({
      _id: {
        $in: payload.fileIds
      }
    });
    if (!files.length) return [];
    return files.reduce(async (lp, file) => {
      const results = await lp;
      const dto = plainToInstance(FileDto, file.toObject());
      const data = await dto.getThumbnail(payload.authenticated, payload.expiresIn);
      results[file._id.toString()] = {
        fileId: file._id,
        ...data
      };
      return results;
    }, {} as any);
  }

  public async downloadPoster(payload: DownloadPosterPayload) {
    const file = await this.FileModel.create({
      name: payload?.filename,
      notification: payload.notification,
      status: FILE_STATUS.PENDING_UPLOAD,
      source: 'local',
      path: payload.filename,
      mediaType: 'image',
      acl: 'authenticated-read',
      createdAt: new Date(),
      updatedAt: new Date()
    });
    await file.save();

    try {
      const request = await axios.get(payload.url, {
        responseType: 'arraybuffer'
      });
      const buffer = Buffer.from(request.data);

      const metadata = await this.imageService.getMetaData(buffer);

      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          format: metadata.format,
          status: FILE_STATUS.PROCESSING,
          updatedAt: new Date()
        }
      });
      // allow max 2k for width
      let width = metadata.width > 2000 ? 2000 : metadata.width;
      let height; // Math.ceil((metadata.height * width) / 2000);
      // overwrite default config if image height is too high
      if (metadata.height > 4000) {
        height = 2000;
        width = undefined;
      }
      if (file.params?.forceImageWidth) {
        width = file.params.forceImageWidth;
        height = file.params.forceImageHeight || undefined;
      } else if (file.params?.forceImageHeight) {
        height = file.params.forceImageHeight;
        width = file.params.forceImageWidth || undefined;
      }
      const newPathBuffer = await this.imageService.resize(buffer, {
        width,
        height,
        // overwrite default quality, should check this option if needed
        quality: 80
      }) as Buffer;
      const fileUpload = this.getFileUploadService('local'); // support local only for now
      const thumbUploadRes = await fileUpload.upload({
        body: newPathBuffer,
        fileType: 'video',
        acl: 'public-read',
        fileId: file._id,
        fileName: `${randomString(5)}_thumb.jpg`,
        deleteLocalFileAfterUploaded: false
      });
      if (thumbUploadRes.path) {
        await this.FileModel.updateOne({ _id: file._id }, {
          $set: {
            thumbnailPath: toPosixPath(thumbUploadRes.path),
            updatedAt: new Date(),
            status: FILE_STATUS.PROCESSED
          }
        });
      }

      const newFile = await this.FileModel.findById(file._id);
      await this.notifyFileProcessed(newFile, 'processed');
    } catch (e) {
      const error = await e;
      await this.FileModel.updateOne({ _id: file._id }, {
        $set: {
          status: FILE_STATUS.ERROR,
          error: error.stack,
          updatedAt: new Date()
        }
      });

      await this.deletePhysicalFiles(file);
      await this.notifyFileProcessed(file, 'error');
    }
  }
}
