import { ReadStream } from 'fs';

import { ObjectId } from 'mongodb';

export interface IFileSize {
  width?: number;
  height?: number;
  // jpg, png
  format?: string;
  // 70, 80...
  quality?: number;
}

export interface IFileUploadOptions {
  /**
   * internal file type such as avatar, feed-image, etc...
   */
  type: string;
  /**
   * who upload
   */
  uploadedBy?: string;
  // options for photos only
  sizes?: IFileSize[]
}

export interface IMulterFileUpload {
  /**
   * multipart/form-data field upload name
   */
  fieldname: string;
  /**
   * eg 'abc.tif'
   */
  originalname: string;
  /**
   * eg '7bit'
   */
  encoding: string;
  /**
   * eg 'image/tiff'
   */
  mimetype: string;
  /**
   * absolute folder name. eg '/absolute/path/public'
   */
  destination: string;
  /**
   * eg 'q1pjq-abc.tif'
   */
  filename: string;
  /**
   * absolute path to the file '/absolute/path/public/q1pjq-abc.tif'
   */
  path: string;
  /**
   * size in bytes
   */
  size: number;
}

export interface IPresignedPostUploadOptions {
  /**
   * file to upload
   */
  key: string;
  /**
   * s3 bucket if any
   */
  bucket?: string;

  /**
   * public-read, authenticated-read. Default is authenticated read
   */
  acl?: string;

  /**
   * PUT or POST. Default is POST
   * with PUT method, we cannot control file name and file size, default is POST
   */
  method?: string;

  /**
   * expires in seconds, defaul 3600
   */
  expiresIn?: number;

  /**
   * max upload size in MB, default is 2048
   */
  maxUploadSizeInMB?: number;

  /**
   * folder to upload
   */
  startWith?: string;
}

export interface IGetSignedUrlOptions {
  /**
   * file to upload
   */
  key: string;
  /**
   * s3 bucket if any
   */
  bucket?: string;

  /**
   * expires in seconds, defaul 15 * 60
   */
  expiresIn?: number;
}

export interface IDeleteObjectOptions {
  /**
   * file to upload
   */
  key: string;
  /**
   * s3 bucket if any
   */
  bucket?: string;
}

export interface IDelteObjectsOptions {
  /**
   * file to upload
   */
  keys: string[];
  /**
   * s3 bucket if any
   */
  bucket?: string;
}

export interface IUploadObjectOptions {
  /**
   * file to upload
   */
  key: string;
  /**
   * s3 bucket if any
   */
  bucket?: string;

  acl: string;

  body: any;

  contentType?: string;
}

export enum EMediaConvertPreset {
  'mp4_240p' = 'mp4_240p',
  'mp4_240p_mobile_vertical' = 'mp4_240p_mobile_vertical',
  'mp4_720p' = 'mp4_720p',
  'mp4_720p_mobile_vertical' = 'mp4_720p_mobile_vertical',
  'mp4_1080p' = 'mp4_1080p',
  'mp4_1080p_mobile_vertical' = 'mp4_1080p_mobile_vertical'
}

export type IMediaConvertPresetValue = `${EMediaConvertPreset}`;


export interface IMediaConvertOptions {
  /**
   * file path in S3 bucket
   */
  input: string;

  /**
   * default ouput folder. Default is `videos/outputs/`
   * Must not have slash at begining. And have slash at the end
   */
  outputDir?: string;

  /**
   *
   */
  presets: IMediaConvertPresetValue[];

  /**
   * from S3 bucket
   */
  fromBucket?: string;

  /**
   * to S3 bucket
   */
  toBucket?: string;

  /**
   * acl to the output file. Default is authenticated-read
   */
  acl?: 'public-read' | 'authenticated-read' | 'PUBLIC_READ' | 'AUTHENTICATED_READ';
}

export interface IVideoStorage {
  /**
   * key file in bucket
   */
  key: string;
  bucket: string;
  url: string;
  acl: 'public-read' | 'authenticated-read';
  type: string;
  preset: string;
  width: string;
  height: string;
  /**
   * mp4, hls, etc...
   */
  format: string;
}

export interface IFileUpload {
  /**
   * file in the local path
   * need to provide localPath or buffer
   */
  localPath?: string;

  /**
   * if it is buffer or read stream
   */
  body?: Buffer | ReadStream;

  /**
   * use if provided
   */
  toDir?: string;

  /**
   * default ACL
   */
  acl: 'public-read' | 'authenticated-read';

  /**
   * delete local file after uploaded
   */
  deleteLocalFileAfterUploaded?: boolean;

  fileId: string | ObjectId;

  /**
   * use this file name
   */
  fileName?: string;

  /**
   * generate random file name. Default true
   */
  generateFileName?: boolean;

  /**
   * file extension to be use
   */
  fileExt?: string;

  fileType: 'image' | 'video' | 'document';
}

export interface IFileUploadResponse {
  absolutePath: string;
  path: string;
}

export interface ICreateVideoThumbnailResponse {
  path: string;
  fileName: string;
}

export interface ICreateVideoThumbnailsOptions {
  filePath: string;

  toDir: string;

  /**
   * array of time to get, otherwise it will get 1 at middle of the video
   * eg 00:00:02
   */
  timemarks?: string[];

  width?: number;

  height?: number;
}

export interface IPublicFileInfo {
  _id: ObjectId;
  url: string;
  thumbnailUrl: string;
  blurredImageUrl: string;
  width: number;
  height: number;
  mediaType: string;
}

export interface PresignedUploadUrl {
  uploadUrl: string;
  bodyBinary: boolean;
  fields: any[];
  fileId: string | ObjectId;
  status: string;
  maxUploadSizeInMB: number;
}
