import { existsSync } from 'fs';

import {
  Injectable,
  NestInterceptor,
  ExecutionContext,
  CallHandler,
  mixin,
  Type
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { mkdirp } from 'mkdirp';
import * as multer from 'multer';
import { Observable } from 'rxjs';

import {
  createAlias, getExt, getFileName, randomString
} from 'src/core';

import { IMulterFileUpload } from '../interfaces';
import { transformException } from '../lib/multer.utils';

export function FileUploadInterceptor(
  fieldName = 'file',
  /**
   * folder to upload
   */
  destination = null
) {
  @Injectable()
  class MixinInterceptor implements NestInterceptor {
    private uploadDir: string;

    constructor(
      private readonly config: ConfigService
    ) {
      this.uploadDir = destination || this.config.get('UPLOAD_DIR');
      this.createFolderIfNotExists(this.uploadDir);
    }

    private createFolderIfNotExists(dir: string) {
      if (!existsSync(dir)) mkdirp.sync(dir);
    }

    async intercept(
      context: ExecutionContext,
      next: CallHandler
    ): Promise<Observable<any>> {
      const ctx = context.switchToHttp();
      const { uploadDir } = this;
      const storage = multer.diskStorage({
        destination(_req, _file, cb) {
          cb(null, uploadDir);
        },
        filename(_req, file, cb) {
          const ext = (getExt(file.originalname) || '').toLocaleLowerCase();
          const orgName = getFileName(file.originalname, true);
          const randomText = randomString(5); // avoid duplicated name, we might check file name first?
          const name = createAlias(`${randomText}-${orgName}`).toLocaleLowerCase() + ext;
          return cb(null, name);
        }
      });
      const upload = multer({ storage }).single(fieldName);
      await new Promise((resolve, reject) => {
        upload(ctx.getRequest(), ctx.getResponse(), (err: any) => {
          if (err) {
            const error = transformException(err);
            // TODO - trace error and throw error
            reject(error);
            return;
          }
          resolve(true);
        });
      });

      const ctxRequest = ctx.getRequest();
      const fileContent: IMulterFileUpload = ctxRequest.file;
      ctxRequest.file = fileContent;

      return next.handle();
    }
  }

  const Interceptor = mixin(MixinInterceptor);
  return Interceptor as Type<NestInterceptor>;
}
