import { Expose, Transform } from 'class-transformer';
import { ObjectId } from 'mongodb';

import { isUrl } from 'src/core';

import { FILE_STATUS } from '../constants';
import { IVideoStorage } from '../interfaces';
import { generateSignedUrlFromPath } from '../lib/file-utils';

export class FileDto {
  @Expose()
  @Transform(({ obj }) => obj._id)
  _id: ObjectId;

  @Expose()
  assetId: string; // ref asset ID

  @Expose()
  jobId: string; // ref asset ID

  @Expose()
  name: string;

  @Expose()
  source: string;

  @Expose()
  type: string; // internal type of file

  @Expose()
  mediaType: string; // video or photo

  @Expose()
  uploadedBy: string;

  @Expose()
  status: string;

  @Expose()
  meta: any;

  @Expose()
  notification: any;

  @Expose()
  size: number;

  @Expose()
  width: number;

  @Expose()
  height: number;

  @Expose()
  bytes: number;

  @Expose()
  format: string;

  @Expose()
  mime: string;

  @Expose()
  version: string;

  @Expose()
  path: string;

  @Expose()
  blurImagePath: string;

  @Expose()
  thumbnailPath: string;

  @Expose()
  url: string;

  @Expose()
  acl: string;

  @Expose()
  error: string;

  @Expose()
  videos: IVideoStorage[];

  @Expose()
  poster: IVideoStorage;

  @Expose()
  createdAt: Date;

  @Expose()
  updatedAt: Date;

  public isPublic(acl = null) {
    return ['public-read', 'PUBLIC_READ'].includes(acl || this.acl);
  }

  public isVideo() {
    return this.mediaType === 'video';
  }

  public isImage() {
    return this.mediaType === 'image';
  }

  public async getThumbnailUrl(authenticated = false, expiresIn = 3600) {
    let thumbnailUrl = null;
    if (this.isPublic() && this.thumbnailPath) {
      if (this.source === 'local') {
        const data = await generateSignedUrlFromPath(this.thumbnailPath, {
          expiresIn,
          excludeHash: true
        });
        thumbnailUrl = data.url;
      }
    } else if (authenticated && this.thumbnailPath) {
      if (this.source === 'local') {
        const data = await generateSignedUrlFromPath(this.thumbnailPath, {
          expiresIn,
          excludeHash: true
        });
        thumbnailUrl = data.url;
      }
      // thumbnailUrl = await S3Service.getSignedUrl({
      //   key: this.thumbnailPath,
      //   // TODO - check me
      //   expiresIn
      // });
    }

    return thumbnailUrl;
  }

  // eslint-disable-next-line
  public async getMainFileUrl(_authenticated = true, expiresIn = 3600, excludeHash = false) {
    if (!this.path) return '';
    if (isUrl(this.path)) return this.path;
    if (this.source === 'local') {
      const data = await generateSignedUrlFromPath(this.path, {
        expiresIn,
        excludeHash
      });
      return data.url;
    }
    // if (this.isPublic() && this.path) url = S3Service.getUrl({ key: this.path });
    // else if (authenticated && this.path) {
    //   url = await S3Service.getSignedUrl({
    //     key: this.path,
    //     // TODO - check me
    //     expiresIn
    //   });
    // }

    return '';
  }

  public async getBlurredImageUrl() {
    if (this.blurImagePath) {
      // return S3Service.getUrl({ key: this.path });
    }

    return null;
  }

  public async getPublicVideoInfo(authenticated = false, expiresIn = 3600) {
    if (this.status === FILE_STATUS.PENDING_UPLOAD) return null;
    const posterUrl = null;
    if (this.poster && this.isPublic()) {
      // posterUrl = S3Service.getUrl({ key: this.poster.key });
    } else if (this.poster && authenticated) {
      // posterUrl = await S3Service.getSignedUrl({
      //   key: this.poster.key,
      //   bucket: this.poster.bucket,
      //   expiresIn
      // });
    }
    const videos = await (this.videos || []).reduce(async (lp, video) => {
      const results = await lp;
      const videoUrl = null;
      if (this.isPublic(video.acl)) {
        // videoUrl = S3Service.getUrl({ key: video.key });
      } else if (authenticated) {
        // videoUrl = await S3Service.getSignedUrl({
        //   key: video.key,
        //   bucket: video.bucket,
        //   expiresIn
        // });
      }

      results.push({
        url: videoUrl,
        width: video.width,
        height: video.height,
        format: video.format
      });
      return results;
    }, [] as any);
    const url = videos.length && videos[0].url ? videos[0].url : await this.getMainFileUrl(authenticated, expiresIn);
    const thumbnailUrl = await this.getThumbnailUrl(authenticated, expiresIn) || posterUrl;
    return {
      _id: this._id,
      url,
      width: this.width,
      height: this.height,
      meta: this.meta,
      thumbnailUrl,
      poster: {
        url: posterUrl || await this.getThumbnailUrl(authenticated, expiresIn)
      },
      videos
    };
  }

  public async getPublicInfo(authenticated = false, exp = 3600, excludeHash = false) {
    const expiresIn = exp || 3600;
    if (this.isVideo()) {
      return this.getPublicVideoInfo(authenticated, expiresIn);
    }

    // with non-authenticated we will respond blur and thumb image only
    const url = await this.getMainFileUrl(authenticated, expiresIn, excludeHash);
    const thumbnailUrl = await this.getThumbnailUrl(authenticated, expiresIn);

    return {
      _id: this._id,
      url,
      thumbnailUrl,
      blurredImageUrl: await this.getBlurredImageUrl(),
      width: this.width,
      height: this.height,
      mediaType: this.mediaType
    };
  }

  public async getPostbackInfo(authenticated = false, expiresIn = 3600) {
    const publicInfo = await this.getPublicInfo(authenticated, expiresIn);
    return {
      _id: this._id,
      acl: this.acl,
      format: this.format,
      type: this.type,
      mediaType: this.mediaType,
      ...publicInfo
    };
  }

  public async getThumbnail(authenticated = false, expiresIn = 3600) {
    const blurredImageUrl = this.blurImagePath ? await this.getBlurredImageUrl() : null;
    let thumbnailUrl = null;
    if (this.thumbnailPath) {
      thumbnailUrl = await this.getThumbnailUrl(authenticated, expiresIn);
    } else if (this.poster) {
      // thumbnailUrl = await S3Service.getSignedUrl({
      //   key: this.poster.key,
      //   bucket: this.poster.bucket,
      //   expiresIn
      // });
    }

    return {
      mediaType: this.mediaType,
      blurredImageUrl,
      thumbnailUrl
    };
  }
}
