import { createHash } from 'crypto';

import { Inject, Injectable } from '@nestjs/common';
import {
  DefaultJobOptions,
  Job,
  Queue, Worker
} from 'bullmq';

import {
  CORE_QUEUE_MESSAGE_REDIS_CONNECTION,
  IRedisQueueParams,
  IQueueJobOptions
} from './constants';
import { QueueWorkerHolderSingleton } from './queue-workers-holder';

@Injectable()
export class QueueService {
  private _queues: Record<string, Queue> = {};

  /**
   * placeholder to store queue and job name
   * issue we have is to change or rename schedule, repeat job and restart application
   * it wont be deleted in redis and will be executed 2 times
   * example
   * 1. run this job: this.queueService.add('test_schedule', 'test_schedule1', { repeat: { pattern: '*\/5 * * * * *'} } as any);
   * 2. stop app while 1 is not done yet (processing but not done job)
   * 3. rename job: this.queueService.add('test_schedule', 'test_schedule2', { repeat: { pattern: '*\/5 * * * * *'} } as any);
   * log test function: console.log('data queue', data.name, data.id); will show 2 lines
   * data queue test_schedule1 repeat:9ee31f9e1d5840816190def2ced7ea47:1718866730000
   * data queue test_schedule2 repeat:470763aae1a95c9e3823804de441e86b:1718866745000
   */
  private _activeQueueAndJobs: Record<string, any> = {};

  private _activeUniqueJobs: string[] = [];

  constructor(
    @Inject(CORE_QUEUE_MESSAGE_REDIS_CONNECTION) private readonly redisQueueParams: IRedisQueueParams
  ) { }

  /**
   * hash md5 and get first 5 characters as new prefix
   * @param str
   * @returns
   */
  private shortenPrefix(str: string, len = 5) {
    return createHash('md5').update(str).digest('hex').substring(0, len);
  }

  private addActiveQueueJob(queueName: string, jobName: string, options: any) {
    this._activeQueueAndJobs[`${queueName}:${jobName}`] = options;
  }

  private addActiveUniqueJobName(queueName: string, jobName: string, options: IQueueJobOptions) {
    if (!options.jobUnique || !options?.jobOptions?.jobId) return;
    const name = `${queueName}:${jobName}:${options?.jobOptions?.jobId}`;

    if (!this._activeUniqueJobs.includes(name)) {
      this._activeUniqueJobs.push(name);
    }
  }

  public getActiveQueueJobs() {
    return { ...this._activeQueueAndJobs };
  }

  public isInActiveQueueJob(job: Job) {
    const keys = this._activeQueueAndJobs.keys();
    return keys.includes(`${job.queueName}:${job.name}`);
  }

  /**
   * check if this unique job is in the active list
   * do not apply with repeat job
   * @param job
   * @returns
   */
  public isInActiveUniqueJob(job: Job) {
    const name = `${job.queueName}:${job.name}:${job.id}`;
    return this._activeUniqueJobs.includes(name);
  }

  /**
   * create new Queue
   * @param name
   * @returns
   */
  public createQueue(name: string, defaultJobOptions?: DefaultJobOptions) {
    const { redisQueueConnection } = this.redisQueueParams;
    return new Queue(name, {
      defaultJobOptions: {
        removeOnComplete: true,
        removeOnFail: {
          // fail will be removed after 1h
          age: 3600,
          count: 10 // max 10 jobs
        },
        ...(defaultJobOptions || {})
      },
      connection: redisQueueConnection,
      // https://docs.bullmq.io/bull/patterns/redis-cluster
      prefix: `{${this.shortenPrefix(name)}}`
    });
  }

  private async removeJobUtilDone(queueName: string, jobId: string) {
    if (!this._queues[queueName]) return true;

    const job = await this._queues[queueName].getJob(jobId);
    if (!job) return true;

    await this._queues[queueName].remove(jobId);
    return new Promise((rs) => {
      setTimeout(async () => {
        rs(this.removeJobUtilDone(queueName, jobId));
      }, 1000);
    });
  }

  /**
   * add queue without create Queue
   * @param queueName
   * @param jobName
   * @param options
   */
  public async add(queueName: string, jobName: string, options?: IQueueJobOptions) {
    if (!this._queues[queueName]) {
      this._queues[queueName] = this.createQueue(queueName, options?.jobOptions);
    }

    // remove previous job if it is all there
    // due to shutdown of process, still have this key in the redis, so job cannot be run
    if (options?.removePreviousJob && options?.jobUnique && options?.jobOptions?.jobId) {
      try {
        // https://github.com/taskforcesh/bullmq/issues/374
        // wait until it is unlocked, up to 30 seconds (default value)
        await this.removeJobUtilDone(queueName, options.jobOptions.jobId);
      } catch {
        // TODO - do something here?
      }
    }

    await this._queues[queueName].add(jobName, options?.data, options?.jobOptions || {});
    this.addActiveQueueJob(queueName, jobName, options);
    this.addActiveUniqueJobName(queueName, jobName, options);
  }

  /**
   *
   * @param queueName handle worker
   * @param handler
   * @param options
   */
  public processWorker(queueName: string, handler, options?: Record<string, any>) {
    const { redisQueueConnection } = this.redisQueueParams;

    const worker = new Worker(queueName, handler, {
      ...(options || {}),
      autorun: false,
      connection: redisQueueConnection,
      // https://docs.bullmq.io/bull/patterns/redis-cluster
      prefix: `{${this.shortenPrefix(queueName)}}`
    });
    worker.run();

    // add placeholder
    QueueWorkerHolderSingleton.addToList(worker);
  }

  public onCompleted(queueName: string, handler) {
    const worker = QueueWorkerHolderSingleton.getByName(queueName);
    if (!worker) throw new Error(`Worker ${queueName} is not ready!`);
    worker.on('completed', handler);
  }

  public onFailed(queueName: string, handler) {
    const worker = QueueWorkerHolderSingleton.getByName(queueName);
    if (!worker) throw new Error(`Worker ${queueName} is not ready!`);
    worker.on('failed', handler);
  }
}
