import { createHash } from 'crypto';

import { Inject, Injectable } from '@nestjs/common';
import { Processor, Queue, Worker } from 'bullmq';

import {
  CORE_QUEUE_MESSAGE_REDIS_CONNECTION,
  QUEUE_SUBSCRIBE_CHANNELS_TOPICS_PREFIX,
  QueueEvent,
  IRedisQueueParams
} from './constants';

@Injectable()
export class QueueMessageService {
  private _queues = {} as Record<string, Queue>;

  private _workers = [] as Array<string>;

  constructor(
    @Inject(CORE_QUEUE_MESSAGE_REDIS_CONNECTION) private readonly redisQueueParams: IRedisQueueParams
  ) { }

  private async getTopicsByChannel(channel: string) {
    const key = `${QUEUE_SUBSCRIBE_CHANNELS_TOPICS_PREFIX}${channel}`;
    const { redisQueueConnection } = this.redisQueueParams;
    const topics = await redisQueueConnection.smembers(key);
    return topics.map((t) => `${channel}_${t}`);
  }

  private async setChannelTopic(channel: string, topic: string) {
    const { redisQueueConnection } = this.redisQueueParams;
    // create channel if not exist

    // add topic to that channel if not exist
    const key = `${QUEUE_SUBSCRIBE_CHANNELS_TOPICS_PREFIX}${channel}`;
    await redisQueueConnection.sadd(key, topic);
  }

  /**
   * hash md5 and get first 5 characters as new prefix
   * @param str
   * @returns
   */
  private shortenPrefix(str: string, len = 5) {
    return createHash('md5').update(str).digest('hex').substring(0, len);
  }

  /**
   *
   * @param channel
   * @param data
   * @param jobName
   * @returns
   */
  async publish(channel: string, data: QueueEvent<any>, jobName = ''): Promise<void> {
    const topics = await this.getTopicsByChannel(channel);
    if (!topics.length) {
      // eslint-disable-next-line no-console
      console.info(`No subscribers in queue message channel ${channel}!`);
      return;
    }

    const { redisQueueConnection, useRedisCluster } = this.redisQueueParams;
    await Promise.all(topics.map((queueName) => {
      const channelQueue = this._queues[queueName] || new Queue(queueName, {
        defaultJobOptions: {
          removeOnComplete: true,
          removeOnFail: 10
        },
        connection: redisQueueConnection,
        // https://docs.bullmq.io/bull/patterns/redis-cluster
        prefix: useRedisCluster ? `{${this.shortenPrefix(queueName)}}` : undefined
      });
      return channelQueue.add(jobName || data.eventName, data);
    }));
  }

  async subscribe(channel: string, topic: string, handler: Processor<any, any, string>, options = {} as any) {
    // TODO - recheck and define if we need to create, define topic / channel list manually
    // avoid key is deleted
    await this.setChannelTopic(channel, topic);
    const { redisQueueConnection, useRedisCluster } = this.redisQueueParams;

    const workerName = `${channel}_${topic}`;
    if (!this._workers.includes(workerName)) {
      this._workers.push(workerName);
      const worker = new Worker(workerName, handler, {
        ...(options || {}),
        autorun: false,
        connection: redisQueueConnection,
        // https://docs.bullmq.io/bull/patterns/redis-cluster
        prefix: useRedisCluster ? `{${this.shortenPrefix(workerName)}}` : undefined
      });
      worker.run();
    }
  }
}
