import { FilterQuery } from 'mongoose';

import { IPopulateDBQuery } from 'src/interfaces';

import { isObjectId, toObjectId } from './string.helper';

export const secondsToTime = (secs: number) => {
  const hours = Math.floor(secs / (60 * 60));

  const dfm = secs % (60 * 60);
  const minutes = Math.floor(dfm / 60);

  const dfs = dfm % 60;
  const seconds = Math.ceil(dfs);

  return {
    h: hours,
    m: minutes,
    s: seconds
  };
};

export const populateDbQuery = (query: Record<string, any>, options: IPopulateDBQuery) => {
  const match = {} as FilterQuery<any>;
  (options.text || []).forEach((k) => {
    if (query[k]) {
      match[k] = { $regex: query[k].trim(), $options: 'i' };
    }
  });

  (options.boolean || []).forEach((k) => {
    if (k in query) {
      if (['false', '0', false, 0].indexOf(query[k]) > -1) {
        match[k] = false;
      } else if (query[k]) {
        match[k] = true;
      }
    }
  });

  (options.equal || []).forEach((k) => {
    if (query[k]) {
      match[k] = query[k];
    }
  });

  (options.objectId || []).forEach((k) => {
    if (query[k] && isObjectId(query[k])) {
      match[k] = toObjectId(query[k]);
    }
  });

  (options.inArray || []).forEach((k) => {
    if (query[k]) {
      match[k] = { $in: query[k].split(',') };
    }
  });

  if (options.excludeId && isObjectId(options.excludeId.toString())) {
    match._id = { $ne: options.excludeId };
  }

  return match;
};

export const populateDBSort = (query: Record<string, any>, defaultSortBy = 'createdAt', defaultSort = -1) => {
  const sort = {};
  if (query.sortBy) {
    sort[query.sortBy] = ['asc', '1'].indexOf(query.sort) > -1 ? 1 : -1;
  } else {
    sort[defaultSortBy] = defaultSort;
  }

  return sort;
};

export const removeNullFromArray = (data: Array<any>) => {
  if (!data?.length) return [];
  return data.filter((d) => !!d);
};
