import { createReadStream, existsSync } from 'fs';
import { basename, extname, join } from 'path';

import {
  BadRequestException,
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Post,
  Put,
  Query,
  Res,
  UseGuards,
  UsePipes,
  ValidationPipe
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

import {
  DataResponse, EntityNotFoundException, ForbiddenException, PageableData, toObjectId
} from 'src/core';
import { CurrentUser } from 'src/decorators';
import { ProfileDto, VideoDto } from 'src/dtos';
import { encryptJwt, verifyHash } from 'src/file-module/lib/file-utils';
import { FileService } from 'src/file-module/services';
import { AuthGuard, LoadProfile } from 'src/guards';
import {
  VideoCreatePayload,
  VideoSearchPayload,
  VideoUpdatePayload
} from 'src/payloads';
import { VideoSearchService, VideoService } from 'src/services';

@Controller('/videos')
export class VideoController {
  constructor(
    private readonly videoService: VideoService,
    private readonly videoSearchService: VideoSearchService,
    private readonly fileService: FileService,
    private readonly configService: ConfigService
  ) { }

  @Get('/search')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(
    @Query() payload: VideoSearchPayload,
    @CurrentUser() user: ProfileDto
  ): Promise<DataResponse<PageableData<VideoDto>>> {
    return DataResponse.ok(await this.videoSearchService.search({ ...payload, status: 'active', verified: true }, user));
  }

  @Get('/uploaded')
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getUploadedVideo(
    @Query() payload: VideoSearchPayload,
    @CurrentUser() user: ProfileDto
  ) {
    return DataResponse.ok(
      await this.videoSearchService.search(
        { ...payload, createdBy: user._id.toString() },
        user
      )
    );
  }

  @Get('/:id/view')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(
    @Param('id') id: string,
    @CurrentUser() user: ProfileDto
  ) {
    return DataResponse.ok(await this.videoService.getDetails(id, user));
  }

  @Post('/main/sign-upload-url')
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(@Body() payload: Record<string, any>) {
    // TODO - define body payload
    const resp = await this.videoService.signUploadMainVideoUrl(payload);
    return DataResponse.ok(resp);
  }

  @Post()
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(
    @Body() payload: VideoCreatePayload,
    @CurrentUser() current: ProfileDto
  ) {
    return DataResponse.ok(await this.videoService.create({ ...payload, createdBy: current._id, isByMember: true }, current));
  }

  @Put('/:id')
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(
    @Param('id') id: string,
    @Body() payload: VideoUpdatePayload,
    @CurrentUser() current: ProfileDto
  ) {
    return DataResponse.ok(
      await this.videoService.update(id, payload, current)
    );
  }

  @Delete('/:id')
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(
    @Param('id') id: string,
    @CurrentUser() current: ProfileDto
  ) {
    return DataResponse.ok(await this.videoService.delete(id, current));
  }

  @Post('/add-view/:id')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async addView(@Param('id') id: string) {
    return DataResponse.ok(await this.videoService.addView(id));
  }

  @Post('/add-share/:id')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async addShare(@Param('id') id: string) {
    return DataResponse.ok(await this.videoService.updateStats(id, { 'stats.shares': 1 }));
  }

  @Get('/download/:id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async download(
    @Param('id') id: string,
    @Res() res: any,
    @Query('token') token: string
  ): Promise<any> {
    if (!token) throw new ForbiddenException();
    if (!verifyHash(token)) throw new ForbiddenException();

    const authenticated = true;// TODO - Check purchased videos for videos for sale or user is admin.
    if (!authenticated) throw new ForbiddenException();

    const video = await this.videoService.findById(id, ['owner']);
    if (!video) throw new EntityNotFoundException();

    const fileInfo = await this.fileService.findById(toObjectId(video.fileId));
    if (!fileInfo) throw new BadRequestException('File not found');

    const uploadDir = this.configService.get('PUBLIC_DIR');
    const filepath = join(uploadDir, fileInfo.path);

    if (!existsSync(filepath)) throw new BadRequestException('File not existed');

    try {
      let name = basename(filepath);
      const ext = extname(name);
      if (!ext) name = `${name}.mp4`;
      res.setHeader('Content-disposition', `attachment; filename=${name}`);
      res.setHeader('Content-Type', fileInfo.mediaType);
      const filestream = createReadStream(filepath);
      filestream.pipe(res);
    } catch {
      throw new BadRequestException();
    }
  }

  @Post('/download/:id')
  @HttpCode(HttpStatus.OK)
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async getVideoDownloadLink(
    @Param('id') id: string
  ): Promise<any> {
    const video = await this.videoService.findById(id);
    if (!video) throw new EntityNotFoundException();

    const authenticated = true;// TODO - Check purchased videos for videos for sale or user is admin.
    if (!authenticated) throw new ForbiddenException();

    const jwt = encryptJwt({
      fileId: video._id.toString()
    });
    return DataResponse.ok(`${process.env.BASE_URL}/videos/download/${id}?token=${jwt}`);
  }
}
