import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { ValidateMongoId } from 'src/core/validators';
import { CurrentUser, Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import {
  VideoCreatePayload, VideoSearchPayload, VideoUpdatePayload
} from 'src/payloads';
import { VideoSearchService, VideoService } from 'src/services';

@Controller('admin/videos')
export class AdminVideoController {
  constructor(
    private readonly videoService: VideoService,
    private readonly videoSearchService: VideoSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: VideoCreatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.videoService.create(payload, current));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: VideoUpdatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.videoService.update(id, payload, current));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.videoService.delete(id, current));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: VideoSearchPayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.videoSearchService.advancedSearch(payload, current));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id', ValidateMongoId) id: string) {
    return DataResponse.ok(await this.videoService.getDetails(id));
  }

  @Post('/thumbnails/signed-upload-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getThumbnailSignedUploadUrl() {
    // TODO - define body payload
    const resp = await this.videoService.signUploadThumbnailUrl();
    return DataResponse.ok(resp);
  }

  @Post('/main/sign-upload-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.videoService.signUploadMainVideoUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.videoService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
