import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser, Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { TagCreatePayload, TagSearchPayload, TagUpdatePayload } from 'src/payloads';
import { TagSearchService, TagService } from 'src/services';

@Controller('admin/tags')
export class AdminTagController {
  constructor(
    private readonly tagService: TagService,
    private readonly tagSearchService: TagSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: TagCreatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.tagService.create(payload, current));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: TagUpdatePayload) {
    return DataResponse.ok(await this.tagService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.tagService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: TagSearchPayload) {
    return DataResponse.ok(await this.tagSearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.tagService.getDetails(id));
  }
}
