import {
  Controller,
  Injectable,
  HttpCode,
  HttpStatus,
  Get,
  Query,
  UseGuards,
  Body,
  Put,
  Param,
  ValidationPipe,
  UsePipes,
  Post
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { Roles } from 'src/decorators';
import { SettingDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { SettingUpdatePayload } from 'src/payloads';
import { SettingService } from 'src/services';

@Injectable()
@Controller('admin/settings')
export class AdminSettingController {
  constructor(private readonly settingService: SettingService) { }

  @Get('')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UseGuards(RoleGuard)
  async getAdminSettings(
    @Query('group') group: string
  ): Promise<DataResponse<SettingDto[]>> {
    const settings = await this.settingService.getEditableSettings(group);
    return DataResponse.ok(settings);
  }

  @Put('/:key')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @UseGuards(RoleGuard)
  async update(
    @Param('key') key: string,
    @Body() value: SettingUpdatePayload
  ): Promise<DataResponse<SettingDto>> {
    const data = await this.settingService.update(key, value);
    return DataResponse.ok(data);
  }

  @Post('/sign-upload-image-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getCsvFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    const resp = await this.settingService.getSignedUploadImageFile(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.settingService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
