import {
  Body,
  Controller, Delete, Get, HttpException, Param, Post, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser } from 'src/decorators';
import { ProfileDto, ReactionDto } from 'src/dtos';
import { AuthGuard } from 'src/guards';
import { ReactDeletePayload, ReactionCreatePayload, ReactionSearchRequestPayload } from 'src/payloads';
import { ReactionSearchService, ReactionService } from 'src/services';

@Controller('/reactions')
export class ReactionController {
  constructor(
    private readonly reactionSearchService: ReactionSearchService,
    private readonly reactionService: ReactionService
  ) { }

  @Post()
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async create(
    @CurrentUser() user: ProfileDto,
    @Body() payload: ReactionCreatePayload
  ): Promise<DataResponse<ReactionDto>> {
    const data = await this.reactionService.create(payload, user);
    return DataResponse.ok(data);
  }

  @Delete()
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async delete(
    @CurrentUser() user: ProfileDto,
    @Body() payload: ReactDeletePayload
  ): Promise<DataResponse<ReactionDto>> {
    const data = await this.reactionService.delete(payload, user);
    return DataResponse.ok(data);
  }

  @Get('/:type/:action')
  @UseGuards(AuthGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(
    @Query() payload: ReactionSearchRequestPayload,
    @CurrentUser() user: ProfileDto,
    @Param() params: Record<string, string>
  ) {
    const { type: objectType, action } = params;
    if (!objectType || !action) {
      return DataResponse.error(new HttpException('Missing params!', 400));
    }
    return DataResponse.ok(await this.reactionSearchService.search({ ...payload, objectType, action }, user));
  }
}
