import {
  Body, Controller, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse, PageableData } from 'src/core';
import { Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { PreventInvalidUsernameGuard } from 'src/guards/prevent-invalid-username.guard';
import {
  AdminProfileUpdatePayload, ProfileAdvancedSearchPayload, ProfileCreatePayload
} from 'src/payloads';
import {
  AuthService, AvatarService, ProfileSearchService, ProfileService
} from 'src/services';

@Controller('/admin/profiles')
export class AdminProfileController {
  constructor(
    private readonly profileService: ProfileService,
    private readonly profileSearchService: ProfileSearchService,
    private readonly authService: AuthService,
    private readonly avatarService: AvatarService
  ) { }

  @Post()
  @UseGuards(RoleGuard, PreventInvalidUsernameGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async adminCreate(
    @Body() payload: ProfileCreatePayload
  ): Promise<DataResponse<ProfileDto>> {
    const profile = await this.profileService.create(payload);

    if (payload.password) {
      await this.authService.createAuthPassword({
        profileId: profile._id,
        value: payload.password
      });
    }

    return DataResponse.ok(profile);
  }

  @Get('/search')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async search(
    @Query() payload: ProfileAdvancedSearchPayload
  ): Promise<DataResponse<PageableData<ProfileDto>>> {
    const data = await this.profileSearchService.advancedSearch(payload);
    return DataResponse.ok(data);
  }

  @Put('/:id')
  @UseGuards(RoleGuard, PreventInvalidUsernameGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async adminUpdate(
    @Body() payload: AdminProfileUpdatePayload,
    @Param('id') profileId: string
  ): Promise<DataResponse<ProfileDto>> {
    const data = await this.profileService.update(profileId, payload);
    return DataResponse.ok(data);
  }

  @Get(['/:id', '/search/:id'])
  @UseGuards(RoleGuard, PreventInvalidUsernameGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async getDetails(
    @Param('id') profileId: string
  ): Promise<DataResponse<ProfileDto>> {
    const data = await this.profileService.getDetails(profileId);
    return DataResponse.ok(data);
  }

  @Post('/avatar/upload-url')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async getDirectUploadAvatarUrl() {
    const res = await this.avatarService.getDirectUploadUrl();
    return DataResponse.ok(res);
  }
}
