import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser, Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { PostCreatePayload, PostSearchPayload, PostUpdatePayload } from 'src/payloads';
import { PostSearchService, PostService } from 'src/services';

@Controller('admin/posts')
export class AdminPostController {
  constructor(
    private readonly postService: PostService,
    private readonly postSearchService: PostSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: PostCreatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.postService.create(payload, current));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: PostUpdatePayload) {
    return DataResponse.ok(await this.postService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.postService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: PostSearchPayload) {
    return DataResponse.ok(await this.postSearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.postService.getDetails(id));
  }

  @Post('/sign-upload-image-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.postService.signUploadImageUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.postService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
