import {
  Controller, Get, Param, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { LoadProfile } from 'src/guards';
import { PerformerSearchPayload } from 'src/payloads';
import { PerformerSearchService, PerformerService } from 'src/services';

@Controller('/performers')
export class PerformerController {
  constructor(
    private readonly performerSearchService: PerformerSearchService,
    private readonly performerService: PerformerService
  ) { }

  @Get('/search')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: PerformerSearchPayload, @CurrentUser() user: ProfileDto) {
    return DataResponse.ok(await this.performerSearchService.search(payload, user));
  }

  @Get('/:id/view')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string, @CurrentUser() user: ProfileDto) {
    return DataResponse.ok(await this.performerService.getDetails(id, user));
  }
}
