import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser, Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { PerformerCreatePayload, PerformerSearchPayload } from 'src/payloads';
import { PerformerUpdatePayload } from 'src/payloads/performer/performer-update.payload';
import { PerformerSearchService, PerformerService } from 'src/services';

@Controller('admin/performers')
export class AdminPerformerController {
  constructor(
    private readonly performerService: PerformerService,
    private readonly performerSearchService: PerformerSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: PerformerCreatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.performerService.create(payload, current));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: PerformerUpdatePayload) {
    return DataResponse.ok(await this.performerService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.performerService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: PerformerSearchPayload) {
    return DataResponse.ok(await this.performerSearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.performerService.getDetails(id));
  }

  @Post('/sign-upload-avatar-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.performerService.getSignedUploadAvatarUrl(payload);
    return DataResponse.ok(resp);
  }

  @Post('/sign-upload-cover-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getCsvFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.performerService.getSignedUploadCoverUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.performerService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
