import {
  Controller,
  Injectable,
  HttpCode,
  HttpStatus,
  UsePipes,
  ValidationPipe,
  Post,
  UseGuards,
  Put,
  Body,
  Param,
  Get,
  Query
} from '@nestjs/common';

import { DataResponse, SearchRequest } from 'src/core';
import { Roles } from 'src/decorators';
import { RoleGuard } from 'src/guards';
import { EmailTemplateUpdatePayload } from 'src/payloads';
import { EmailTemplateService, MailService } from 'src/services';

@Injectable()
@Controller('mailer')
export class MailerController {
  constructor(private readonly mailService: MailService, private readonly emailTemplateService: EmailTemplateService) { }

  @Post('/verify')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UseGuards(RoleGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async verify(): Promise<DataResponse<any>> {
    const data = await this.mailService.verify();
    return DataResponse.ok(data);
  }

  @Put('/templates/:id')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UseGuards(RoleGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async update(
    @Body() payload: EmailTemplateUpdatePayload,
    @Param('id') id: string
  ): Promise<DataResponse<any>> {
    const data = await this.emailTemplateService.updateTemplate(id, payload);
    return DataResponse.ok(data);
  }

  @Get('/templates')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UseGuards(RoleGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async getAll(@Query() payload: SearchRequest): Promise<DataResponse<any>> {
    const data = await this.emailTemplateService.getAllTemplates(payload);
    return DataResponse.ok(data);
  }

  @Get('/templates/:id')
  @HttpCode(HttpStatus.OK)
  @Roles('admin')
  @UseGuards(RoleGuard)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async findOne(
    @Param('id') id: string
  ): Promise<DataResponse<any>> {
    const data = await this.emailTemplateService.findById(id);
    return DataResponse.ok(data);
  }
}
