import {
  Controller, Get, Param, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { LoadProfile } from 'src/guards';
import { ChannelSearchPayload } from 'src/payloads/channel';
import { ChannelSearchService, ChannelService } from 'src/services/channel';

@Controller('/channels')
export class ChannelController {
  constructor(
    private readonly channelSearchService: ChannelSearchService,
    private readonly channelService: ChannelService
  ) { }

  @Get('/search')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: ChannelSearchPayload, @CurrentUser() user: ProfileDto) {
    return DataResponse.ok(await this.channelSearchService.search(payload, user));
  }

  @Get('/:id/view')
  @UseGuards(LoadProfile)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string, @CurrentUser() user: ProfileDto) {
    return DataResponse.ok(await this.channelService.getDetails(id, user));
  }
}
