import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { CurrentUser, Roles } from 'src/decorators';
import { ProfileDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import { ChannelCreatePayload, ChannelSearchPayload, ChannelUpdatePayload } from 'src/payloads/channel';
import { ChannelSearchService, ChannelService } from 'src/services/channel';


@Controller('admin/channels')
export class AdminChannelController {
  constructor(
    private readonly channelService: ChannelService,
    private readonly channelSearchService: ChannelSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: ChannelCreatePayload, @CurrentUser() current: ProfileDto) {
    return DataResponse.ok(await this.channelService.create(payload, current));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: ChannelUpdatePayload) {
    return DataResponse.ok(await this.channelService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.channelService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: ChannelSearchPayload) {
    return DataResponse.ok(await this.channelSearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.channelService.getDetails(id));
  }

  @Post('/sign-upload-image-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.channelService.signUploadImageUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.channelService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
