import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { Roles } from 'src/decorators';
import { RoleGuard } from 'src/guards';
import { CategoryCreatePayload, CategorySearchRequest, CategoryUpdatePayload } from 'src/payloads';
import { CategorySearchService, CategoryService } from 'src/services';

@Controller('admin/categories')
export class AdminCategoryController {
  constructor(
    private readonly categoryService: CategoryService,
    private readonly categorySearchService: CategorySearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: CategoryCreatePayload) {
    return DataResponse.ok(await this.categoryService.create(payload));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: CategoryUpdatePayload) {
    return DataResponse.ok(await this.categoryService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.categoryService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: CategorySearchRequest) {
    return DataResponse.ok(await this.categorySearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.categoryService.getDetails(id));
  }

  @Post('/sign-upload-poster-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.categoryService.signUploadPosterUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.categoryService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
