import {
  Body, Controller, Post, Put, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse, EntityNotFoundException } from 'src/core';
import { Roles } from 'src/decorators';
import { AuthUpdateDto } from 'src/dtos';
import { RoleGuard } from 'src/guards';
import {
  AdminChangePasswordPayload, ForgotPayload
} from 'src/payloads';
import { AuthService, MailService, ProfileService } from 'src/services';

@Controller('/auth')
export class PasswordController {
  constructor(
    private authService: AuthService,
    private readonly mailService: MailService,
    private readonly profileService: ProfileService
  ) { }

  @Put('/admin/change-password')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async adminUpdatePassword(
    @Body() payload: AdminChangePasswordPayload
  ) {
    const profile = await this.profileService.findById(payload.profileId);
    if (!profile) throw new EntityNotFoundException();

    await this.authService.updateAuthPassword(
      new AuthUpdateDto({
        type: 'password',
        profileId: payload.profileId,
        value: payload.password
      })
    );

    if (profile.email) {
      await this.mailService.sendMail({
        template: 'admin-change-password',
        to: profile.email,
        data: {
          password: payload.password
        }
      });
    }


    return DataResponse.ok(true);
  }

  @Post('/forgot')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  async forgot(
    @Body() payload: ForgotPayload
  ): Promise<DataResponse<{ success: boolean }>> {
    await this.authService.forgot(payload.email);

    return DataResponse.ok({
      success: true
    });
  }
}
