import {
  Body, Controller, Delete, Get, Param, Post, Put, Query, UseGuards, UsePipes, ValidationPipe
} from '@nestjs/common';

import { DataResponse } from 'src/core';
import { Roles } from 'src/decorators';
import { RoleGuard } from 'src/guards';
import { AdCreatePayload, AdSearchPayload, AdUpdatePayload } from 'src/payloads';
import { AdSearchService, AdService } from 'src/services';

@Controller('admin/ads')
export class AdminAdController {
  constructor(
    private readonly adService: AdService,
    private readonly adSearchService: AdSearchService
  ) { }

  @Post()
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async create(@Body() payload: AdCreatePayload) {
    return DataResponse.ok(await this.adService.create(payload));
  }

  @Put('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async update(@Param('id') id: string, @Body() payload: AdUpdatePayload) {
    return DataResponse.ok(await this.adService.update(id, payload));
  }

  @Delete('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async delete(@Param('id') id: string) {
    return DataResponse.ok(await this.adService.delete(id));
  }

  @Get('/')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async search(@Query() payload: AdSearchPayload) {
    return DataResponse.ok(await this.adSearchService.advancedSearch(payload));
  }

  @Get('/:id')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getDetails(@Param('id') id: string) {
    return DataResponse.ok(await this.adService.getDetails(id));
  }

  @Post('/sign-upload-file-url')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getMainFileSignedUploadUrl(
    @Body() payload: Record<string, any>
  ) {
    // TODO - define body payload
    const resp = await this.adService.signUploadFileUrl(payload);
    return DataResponse.ok(resp);
  }

  @Get('/files/:fileId/info')
  @UseGuards(RoleGuard)
  @Roles('admin')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  public async getFileUploadedInfo(
    @Param('fileId') fileId: string
  ) {
    const resp = await this.adService.getFileUploadedInfo(fileId);
    return DataResponse.ok(resp);
  }
}
