import { join } from 'path';

import { HttpModule } from '@nestjs/axios';
import {
  MiddlewareConsumer, Module, NestModule
} from '@nestjs/common';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { MongooseModule } from '@nestjs/mongoose';
import { ServeStaticModule } from '@nestjs/serve-static';

import { AppController } from './app.controller';
import authConfig from './config/auth';
import configuration from './config/configuration';
import {
  AdminSettingController,
  AdminVideoController,
  LoginController,
  ProfileController,
  AvatarController,
  SettingController,
  VideoController,
  AdminPerformerController,
  PerformerController,
  AdminProfileController,
  AdminTagController,
  TagController,
  AdminPostController,
  PostController,
  ReactionController,
  PasswordController,
  AdminCategoryController,
  CategoryController,
  MenuController,
  LanguageController,
  MailerController,
  StatsController
} from './controllers';
import { AdController, AdminAdController } from './controllers/ad';
import { AdminChannelController, ChannelController } from './controllers/channel';
import { CoreQueueModule } from './core';
import { RequestLoggerMiddleware } from './core/logger/request-log.middleware';
import { RequestLog, RequestLogSchema } from './core/logger/request-log.schema';
import { FileModule } from './file-module/file.module';
import { DownloadPosterFormTC } from './jobs';
import {
  CategoryListener,
  ReactionListener, UpdateVideoConvertStatusListener, VideoListener, ProfileListener,
  PerformerListener,
  VideoTagListener,
  ReactionVideoListener,
  ChannelListener,
  AdsListener
} from './listeners';
import {
  Auth,
  AuthSchema,
  Category,
  CategorySchema,
  Profile,
  ProfileSchema,
  Setting,
  SettingSchema,
  Video,
  VideoSchema,
  Performer,
  PerformerSchema,
  ProfileVerification,
  ProfileVerificationSchema,
  Tag,
  TagSchema,
  Post,
  PostSchema,
  Reaction,
  ReactionSchema,
  EmailTemplate,
  EmailTemplateSchema,
  ForgotPassword,
  ForgotPasswordSchema,
  Stat,
  StatSchema,
  ChannelSchema,
  Channel,
  AdSchema,
  Ad,
  VideoAdsSchema,
  VideoAds
} from './schemas';
import {
  AuthService,
  ProfileService,
  ProfileSearchService,
  VideoService,
  VideoSearchService,
  SettingService,
  PerformerService,
  PerformerSearchService,
  ProfileVerificationService,
  MailService,
  CategoryService,
  TagService,
  TagSearchService,
  PostSearchService,
  PostService,
  ReactionSearchService,
  ReactionService,
  PerformerReactionService,
  ReactionObjectService,
  VideoReactionService,
  CategorySearchService,
  AvatarService,
  LanguageService,
  EmailTemplateService,
  StatsService,
  AdSearchService,
  AdService
} from './services';
import { ChannelReactionService, ChannelSearchService, ChannelService } from './services/channel';

const services = [
  AuthService,
  ProfileService,
  ProfileSearchService,
  CategoryService,
  VideoService,
  ChannelService,
  ChannelSearchService,
  ChannelReactionService,
  VideoSearchService,
  SettingService,
  PerformerService,
  PerformerSearchService,
  ProfileVerificationService,
  MailService,
  TagService,
  TagSearchService,
  PostSearchService,
  PostService,
  ReactionSearchService,
  ReactionService,
  ReactionObjectService,
  PerformerReactionService,
  VideoReactionService,
  CategorySearchService,
  CategoryService,
  AvatarService,
  LanguageService,
  EmailTemplateService,
  StatsService,
  AdSearchService,
  AdService
];

const listeners = [
  UpdateVideoConvertStatusListener,
  ReactionListener,
  VideoListener,
  CategoryListener,
  ProfileListener,
  AdsListener,
  PerformerListener,
  VideoTagListener,
  ReactionVideoListener,
  ChannelListener
];

const jobs = [
  DownloadPosterFormTC
];

@Module({
  imports: [
    ConfigModule.forRoot({
      isGlobal: true,
      load: [configuration, authConfig]
    }),
    CoreQueueModule.registerAsync({
      imports: [ConfigModule],
      useFactory: async (configService: ConfigService) => ({
        redisConfig: configService.get('REDIS_QUEUE_CONFIG'),
        useRedisCluster: configService.get('REDIS_QUEUE_USE_CLUSTER_MODE')
      }),
      inject: [ConfigService]
    }),
    HttpModule.register({
      timeout: 10000,
      maxRedirects: 5
    }),
    MongooseModule.forRootAsync({
      imports: [ConfigModule],
      useFactory: async (configService: ConfigService) => ({
        uri: configService.get<string>('MONGODB_URI')
      }),
      inject: [ConfigService]
    }),
    MongooseModule.forFeature([
      { name: Auth.name, schema: AuthSchema },
      { name: Profile.name, schema: ProfileSchema },
      { name: Video.name, schema: VideoSchema },
      { name: Setting.name, schema: SettingSchema },
      { name: Category.name, schema: CategorySchema },
      { name: Performer.name, schema: PerformerSchema },
      { name: ProfileVerification.name, schema: ProfileVerificationSchema },
      { name: Tag.name, schema: TagSchema },
      { name: Post.name, schema: PostSchema },
      { name: Reaction.name, schema: ReactionSchema },
      { name: EmailTemplate.name, schema: EmailTemplateSchema },
      { name: RequestLog.name, schema: RequestLogSchema },
      { name: ForgotPassword.name, schema: ForgotPasswordSchema },
      { name: Stat.name, schema: StatSchema },
      { name: VideoAds.name, schema: VideoAdsSchema },
      { name: Channel.name, schema: ChannelSchema },
      { name: Ad.name, schema: AdSchema }
    ]),
    ServeStaticModule.forRoot({
      rootPath: join(__dirname, '..', 'public')
    }),
    FileModule
  ],
  controllers: [
    AppController,
    LoginController,
    ProfileController,
    AdminAdController,
    AvatarController,
    VideoController,
    AdminVideoController,
    AdminSettingController,
    SettingController,
    PerformerController,
    AdminChannelController,
    ChannelController,
    AdminPerformerController,
    AdController,
    AdminProfileController,
    AdminTagController,
    TagController,
    AdminPostController,
    PostController,
    ReactionController,
    PasswordController,
    AdminCategoryController,
    CategoryController,
    MenuController,
    LanguageController,
    MailerController,
    StatsController
  ],
  providers: [...services, ...listeners, ...jobs]
})
export class AppModule implements NestModule {
  constructor(private settingService: SettingService) {
    this.settingService.syncCache();
  }

  configure(consumer: MiddlewareConsumer) {
    consumer.apply(RequestLoggerMiddleware);
  }
}
