const { DB, COLLECTION } = require('../migrations/lib');

const SETTING_KEYS = {
  SITE_NAME: 'siteName',
  LOGO_URL: 'logoUrl',
  FAVICON: 'favicon',
  ADMIN_EMAIL: 'adminEmail',
  SENDER_EMAIL: 'senderEmail',
  HOME_TITLE: 'homeTitle',
  HOME_META_KEYWORDS: 'homeMetaKeywords',
  HOME_META_DESCRIPTION: 'homMetaDescription',
  SITE_COPYRIGHT: 'siteCopyright',
  SINGUP_PLACEHODER_IMAGE: 'signupPlacehoderImage',
  DEFAULT_THUMBNAIL_IMAGE: 'defaultThumbnailImage',
  FOOTER_CONTENT: 'footerContent',

  HEADER_SCRIPT: 'headerScript',
  AFTER_BODY_SCRIPT: 'afterBodyScript',
  HTML_META_TAGS: 'htmlMetaTags',
  MONTHLY_SUBSCRIPTION_COMMISSION: 'monthlySubscriptionCommission',
  YEARLY_SUBSCRIPTION_COMMISSION: 'yearlySubscriptionCommission',
  VIDEO_SALE_COMMISSION: 'videoSaleCommission',

  PAYGATE_SUPPORT: 'paygateSupport',

  CCBILL_CLIENT_ACCOUNT_NUMBER: 'ccbillClientAccountNumber',
  CCBILL_SINGLE_SUB_ACCOUNT_NUMBER: 'ccbillSingleSubAccountNumber',
  CCBILL_SUBSCRIPTION_SUB_ACCOUNT_NUMBER: 'ccbillSubsciptionSubAccountNumber',
  CCBILL_FLEXFORM_ID: 'ccbillFlexformId',
  CCBILL_SALT: 'ccbillSalt',
  CCBILL_DATALINK_USERNAME: 'ccbillDatalinkUsername',
  CCBILL_DATALINK_PASSWORD: 'ccbillDatalinkPassword',

  GOOGLE_ANALYTICS_CODE: 'gaCode',
  GOOGLE_TAG_MANAGER_ID: 'gtmId',

  SMTP_USERNAME: 'smtpUsername',
  SMTP_PASSWORD: 'smtpPassword',
  SMTP_HOST: 'smtpHost',
  SMTP_PORT: 'smtpPort',

  ENABLE_MAINTENANCE_MODE: 'enableMaintenanceMode',

  DEFAULT_LOCALE: 'defaultLocale',
  LOCALES: 'locales',
  ENABLE_MULTILANGUAGE: 'enableMultiLanguague'
};

const generalSettings = [
  {
    key: SETTING_KEYS.SITE_NAME,
    value: process.env.SITE_NAME || 'xStreamer',
    name: 'Site name',
    description: 'Global name',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    ordering: 0
  },
  {
    key: SETTING_KEYS.LOGO_URL,
    value: '',
    name: 'Site logo',
    description: 'Recommended size 320x240 pixels. Max file size 5MB. Accept image/*',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    meta: {
      upload: true,
      image: true
    },
    ordering: 1
  },
  {
    key: SETTING_KEYS.FAVICON,
    value: '',
    name: 'Favicon',
    description: 'Recommended size 75x75 pixels. Max file size 5MB. Accept image/*',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    meta: {
      upload: true,
      image: true
    },
    ordering: 2
  },
  // {
  //   key: SETTING_KEYS.PAYGATE_SUPPORT,
  //   value: ['ccbill'],
  //   name: 'Payment supports',
  //   description: 'Select supported paygate in the system',
  //   public: true,
  //   autoload: false,
  //   group: 'general',
  //   editable: true,
  //   meta: {
  //     options: [{
  //       key: 'ccbill',
  //       value: 'ccbill',
  //       label: 'CCBill'
  //     }],
  //     mode: 'multiple'
  //   },
  //   type: 'select',
  //   ordering: 5
  // },
  // {
  //   key: SETTING_KEYS.SINGUP_PLACEHODER_IMAGE,
  //   value: '',
  //   name: 'Signup Placeholder',
  //   description: 'Recommended size 640x480 pixels. Max file size 5MB. Accept image/*',
  //   public: true,
  //   autoload: true,
  //   group: 'general',
  //   editable: true,
  //   meta: {
  //     upload: true,
  //     image: true
  //   },
  //   ordering: 6
  // },
  {
    key: SETTING_KEYS.DEFAULT_THUMBNAIL_IMAGE,
    value: '',
    name: 'Default Thumbnail',
    description: 'Recommended size 320x240 pixels. Max file size 5MB. Accept image/*',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    meta: {
      upload: true,
      image: true
    },
    ordering: 7
  },
  {
    key: SETTING_KEYS.FOOTER_CONTENT,
    value: `<p style="text-align:center;">
      <strong>${process.env.DOMAIN} © Copyright 2024. All Rights Reserved</strong>
    </p>
    <p style="text-align:center;">
      <img src="https://www.dmca.com/img/dmca_logo.png?=sd" alt="dcma" style="height: 35px;width: 100px;display:inline-block;margin:0 auto"/>
    </p>`,
    name: 'Footer content',
    description: 'Add texts for your footer here',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    visible: true,
    type: 'text-editor',
    ordering: 8
  },
  {
    key: SETTING_KEYS.ENABLE_MAINTENANCE_MODE,
    value: false,
    name: 'Website Maintenance',
    description: '',
    public: true,
    autoload: true,
    group: 'general',
    editable: true,
    ordering: 20,
    type: 'boolean'
  }
];

const emailSettings = [
  {
    key: SETTING_KEYS.ADMIN_EMAIL,
    value: process.env.ADMIN_EMAIL || `admin@${process.env.DOMAIN}`,
    name: 'Admin email',
    description: 'Email will receive information from site features by default.',
    public: false,
    group: 'email',
    editable: true,
    ordering: 0,
    autoload: false
  },
  {
    key: SETTING_KEYS.SENDER_EMAIL,
    value: process.env.SENDER_EMAIL || `noreply@${process.env.DOMAIN}`,
    name: 'Sender email',
    description: 'The sender email is the email address that would send your notification or autoresponder email, you can use format [name] <email-address> if want to use name (eg Support <noreply@example.com>). Remember it must be valid with SMTP account',
    public: false,
    group: 'email',
    editable: true,
    ordering: 1,
    autoload: false
  }
];

const customSeoSettings = [
  {
    key: SETTING_KEYS.HOME_TITLE,
    value: 'Home',
    name: 'Home title',
    description: 'Custom your home title',
    public: true,
    group: 'seo',
    editable: true,
    ordering: 0,
    autoload: true
  },
  {
    key: SETTING_KEYS.HOME_META_KEYWORDS,
    value: '',
    name: 'Home meta keywords',
    description: 'Custom meta keywords for home page',
    public: true,
    group: 'seo',
    editable: true,
    ordering: 1,
    autoload: true
  },
  {
    key: SETTING_KEYS.HOME_META_DESCRIPTION,
    value: 'xStreamer',
    name: 'Home meta description',
    description: 'Custom meta description for home page. Should less than 200 characters',
    public: true,
    group: 'seo',
    editable: true,
    type: 'text',
    ordering: 2,
    autoload: true
  }
];

const customScriptSettings = [
  {
    key: SETTING_KEYS.HEADER_SCRIPT,
    value: '',
    name: 'Custom head script',
    description: 'Custom code at <head> tag will be added to all pages',
    public: true,
    autoload: true,
    group: 'customScript',
    editable: true,
    type: 'textarea',
    ordering: 2
  },
  {
    key: SETTING_KEYS.AFTER_BODY_SCRIPT,
    value: '',
    name: 'Custom body script',
    description: 'Custom code at end of <body> tag, will be added to all pages',
    public: true,
    autoload: true,
    group: 'customScript',
    editable: true,
    type: 'textarea',
    ordering: 3
  },
  {
    key: SETTING_KEYS.HTML_META_TAGS,
    value: JSON.stringify({ 'google-site-verification': 'fdcbe01a61a7e9ed81416704b507ff40' }),
    name: 'HTML Meta Tags',
    description: 'Add metadata tag here. Accept JSON only',
    public: true,
    autoload: true,
    group: 'customScript',
    editable: true,
    type: 'textarea',
    ordering: 4
  },
  {
    key: SETTING_KEYS.GOOGLE_ANALYTICS_CODE,
    value: '',
    name: 'GA code',
    description: 'Google Analytics Code eg: GA-123456xx. Enter if have',
    public: true,
    autoload: true,
    group: 'customScript',
    editable: true,
    type: 'text'
  },
  {
    key: SETTING_KEYS.GOOGLE_TAG_MANAGER_ID,
    value: '',
    name: 'Google Tag Manager',
    description: 'Google Tag Manager ID eg: GTM-XYZxx. Enter if have',
    public: true,
    autoload: true,
    group: 'customScript',
    editable: true,
    type: 'text'
  }
];

const commissionSettings = [
  {
    key: SETTING_KEYS.MONTHLY_SUBSCRIPTION_COMMISSION,
    value: 0.2,
    name: 'Monthly subscription commission',
    description: 'Value is from 0.01-0.99 (1%-99%)',
    public: false,
    group: 'commission',
    editable: true,
    type: 'number',
    ordering: 0
  },
  {
    key: SETTING_KEYS.YEARLY_SUBSCRIPTION_COMMISSION,
    value: 0.2,
    name: 'Yearly subscription commission',
    description: '0.1 means that admin will take 10% & model take 90% from total price',
    public: false,
    group: 'commission',
    editable: true,
    type: 'number',
    ordering: 1
  },
  {
    key: SETTING_KEYS.VIDEO_SALE_COMMISSION,
    value: 0.2,
    name: 'Video commission',
    description: 'Video for sale commission',
    public: false,
    group: 'commission',
    editable: true,
    type: 'number',
    ordering: 2
  }
];

const smtpSettings = [
  {
    key: SETTING_KEYS.SMTP_HOST,
    value: '',
    name: 'SMTP Host',
    description: 'Enter host URl of SMTP. eg smtp.example.com',
    public: false,
    group: 'smtp',
    editable: true,
    type: 'text',
    ordering: 0
  },
  {
    key: SETTING_KEYS.SMTP_PORT,
    value: 25,
    name: 'SMTP port',
    description: 'Enter SMTP port: 25, 465, 587...',
    public: false,
    group: 'smtp',
    editable: true,
    type: 'number',
    ordering: 0
  },
  {
    key: SETTING_KEYS.SMTP_USERNAME,
    value: '',
    name: 'SMTP username',
    description: 'Enter authentication username',
    public: false,
    group: 'smtp',
    editable: true,
    type: 'text',
    ordering: 0
  },
  {
    key: SETTING_KEYS.SMTP_PASSWORD,
    value: '',
    name: 'SMTP password',
    description: 'Enter authentication password',
    public: false,
    group: 'smtp',
    editable: true,
    type: 'text',
    ordering: 0
  }
];

module.exports = async () => {
  // eslint-disable-next-line no-console
  console.log('Migrate default settings');

  const settings = [
    ...generalSettings,
    ...emailSettings,
    ...customSeoSettings,
    ...customScriptSettings,
    ...commissionSettings,
    ...smtpSettings
  ];

  await settings.reduce(async (lp, setting) => {
    await lp;

    const checkKey = await DB.collection(COLLECTION.SETTING).findOne({
      key: setting.key
    });
    if (!checkKey) {
      await DB.collection(COLLECTION.SETTING).insertOne({
        ...setting,
        type: setting.type || 'text',
        createdAt: new Date(),
        updatedAt: new Date()
      });
      // eslint-disable-next-line no-console
      console.log(`Inserted setting: ${setting.key}`);
    } else {
      // eslint-disable-next-line no-console
      console.log(`Setting: ${setting.key} exists`);
    }

    return Promise.resolve();
  }, Promise.resolve());

  // eslint-disable-next-line no-console
  console.log('Migrate settings done');
};
